# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import logging

from qb2.api.v1 import (
    extractors as qe,
    filters as qf
)


class DeterrenceDistributionMetricProvider(object):
    def __init__(self, config):
        super(DeterrenceDistributionMetricProvider, self).__init__()
        self.config = config
        self.logger = logging.getLogger(__name__)

    def get(self, name, source):
        self.logger.info('Extracting data for source {}'.format(name))
        extractors = {
            'bs_watch_log': self.__get_bs_watch_log_extract,
            'atom_cube': self.__get_atom_cube_extract
        }
        extractor = extractors[name]
        return extractor(source, self.config[name])

    def __get_bs_watch_log_extract(self, source, config):
        def get_plaform_os(raw_platform, parsed_os):
            platform_hints = set([
                (parsed_os.get(os_trait) or '').lower()
                for os_trait in ['OSFamily', 'OSName']
            ])
            if raw_platform == 'touch':
                if 'android' in platform_hints:
                    return 'touch_android'
                if 'ios' in platform_hints:
                    return 'touch_ios'
            return raw_platform

        return source.qb2(
            log='bs-watch-log',
            fields=[
                'uid', 'canonized_vhost', 'domain',
                'timestamp', 'page', 'counter_id', 'parsed_os',
                qe.custom(
                    'service',
                    lambda counter_id: config['services_codes'].get(counter_id, '-')),
                qe.custom(
                    'raw_platform',
                    lambda counter_id: config['platforms_codes'].get(counter_id, '-'))
            ],
            filters=[
                qf.default_filtering('bs-watch-log'),
                qf.defined('uid', 'parsed_os'),
                qf.one_of('counter_id', set(config['white_list_counter_ids'])),
            ]
        ).qb2(
            log='generic-yson-log',
            fields=[
                qe.log_field('service'),
                qe.log_field('timestamp').rename('unixtime'),
                qe.log_field('uid').rename('yandexuid'),
                qe.log_field('raw_platform').hide(),
                qe.log_field('parsed_os').hide(),
                qe.custom('platform', get_plaform_os, 'raw_platform', 'parsed_os')
            ]
        )

    def __get_atom_cube_extract(self, source, config):
        def get_distr_obj(raw_distr_obj, referer):
            suffices = {
                'yandex.ru': '_morda',
                'yandex.ru/search': '_serp'
            }
            if raw_distr_obj == 'portal_popup':
                raw_distr_obj += suffices.get(referer, '')
            return raw_distr_obj

        return source.qb2(
            log='generic-yson-log',
            fields=[
                qe.log_field('yandexuid'),
                qe.log_field('unixtime'),
                qe.log_field('distr_obj').rename('raw_distr_obj').hide(),
                qe.log_field('eventtype').hide(),
                qe.log_field('past').hide(),
                qe.log_field('referer'),
                qe.custom('distr_obj', get_distr_obj, 'raw_distr_obj', 'referer')
            ],
            filters=[
                qf.defined('raw_distr_obj', 'referer'),
                qf.equals('eventtype', 'show'),
                qf.not_(qf.nonzero('past')),
                qf.one_of('raw_distr_obj', set(config['white_list_distr_objs'])),
                qf.one_of('referer', set(config['white_list_referers']))
            ],
        )
