# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import io
import operator
import logging

from .mail_pusher import MailPusher
from ..utils.tools import safe_divide


class MailDistributionAlertsPusher(MailPusher):
    def __init__(self, config, do_push=True):
        super(MailDistributionAlertsPusher, self).__init__(do_push=do_push)
        self.logger = logging.getLogger(__name__)
        self.config = config
        self.do_push = do_push

    def push(self, alerts):
        self.logger.info('Sending {} alerts to {}'.format(len(alerts), self.config['send_to']))
        if alerts:
            self._push_internal(
                send_from=self.config['send_from'],
                send_to=self.config['send_to'],
                subject='Alert for v4_daily report',
                text=self.__build_report(alerts),
                file_paths=[],
                server='localhost'
            )

    def __build_report(self, alerts):
        stream = io.BytesIO()
        stream.write('Hi! Today {} alert(s) are waiting for your review.\n'.format(len(alerts)))
        for alert in sorted(alerts, key=lambda alert: '\t'.join(map(operator.itemgetter(0), alert['slice']))):
            self.logger.info('Sending alert: {}'.format(alert))
            stream.write('event       : {}\n'.format(alert['event']))
            stream.write('factor      : {}\n'.format(safe_divide(float(alert['count_now']), alert['count_then'])))
            stream.write('date_then   : {}\n'.format(alert['date_then']))
            stream.write('date_now    : {}\n'.format(alert['date_now']))
            stream.write('count_then  : {}\n'.format(alert['count_then']))
            stream.write('count_now   : {}\n'.format(alert['count_now']))

            for dimension, value in alert['slice']:
                if dimension in alert['dimensions']:
                    stream.write('{:11} : {}\n'.format(dimension, value))
            url = self.__build_url([alert['event']], alert['slice'], alert['date_then'], alert['date_now'])
            stream.write('{:11} : {}\n'.format('url', url))
        text = stream.getvalue()
        stream.close()
        return text

    def __build_url(self, events, slice, date_min, date_max, scale='d'):
        url = '{}{}'.format(self.config['ya_stat_url'], self.config['project_path'])
        url += '?scale={}'.format(scale)
        for event in events:
            url += '&_incl_fields={}'.format(event)
        for dimension, value in slice:
            url += '&{}={}'.format(dimension, value)
        url += '&date_min={}+00%3A00%3A00'.format(date_min)
        url += '&date_max={}+00%3A00%3A00'.format(date_max)
        return url
