# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import logging

import os
import smtplib
import email
from email import Encoders
from email.MIMEBase import MIMEBase
from email.MIMEText import MIMEText
from email.MIMEMultipart import MIMEMultipart
from email.Utils import COMMASPACE, formatdate


class MailPusher(object):
    def __init__(self, do_push=True):
        super(MailPusher, self).__init__()
        self.logger = logging.getLogger(__name__)
        self.do_push = do_push

    def push(self, send_from, send_to, subject, text, file_paths=None, server='localhost'):
        self._push_internal(send_from, send_to, subject, text, file_paths or [], server)

    def _push_internal(self, send_from, send_to, subject, text, file_paths, server):
        try:
            self.logger.info('Sending email from server {}.'.format(server))
            self.logger.info('Sender: {}, recipients: {}, text size: {}, files attached {}'.format(
                send_from, send_to, len(text), len(file_paths)
                )
            )
            if self.do_push:
                self.__send_mail(send_from, send_to, subject, text, file_paths, server)
        except (email.errors.MessageError, smtplib.SMTPException) as error:
            self.logger.critical('Failed to send email to {}. Error: {}'.format(send_to, error))

    def __send_mail(self, send_from, send_to, subject, text, file_paths, server):
        message = MIMEMultipart()
        message['From'] = send_from
        message['To'] = COMMASPACE.join(send_to)
        message['Date'] = formatdate(localtime=True)
        message['Subject'] = subject
        message.attach(MIMEText(text.encode('utf8')))

        for file_path in file_paths:
            part = MIMEBase('application', "octet-stream")
            with open(file_path, "rb") as binary_file:
                part.set_payload(binary_file.read())
            Encoders.encode_base64(part)
            part.add_header(
                'Content-Disposition',
                'attachment; filename="{}"'.format(os.path.basename(file_path))
            )
            message.attach(part)

        server = smtplib.SMTP(server)
        try:
            server.sendmail(send_from, send_to, message.as_string())
        finally:
            server.close()
