# -*- coding: utf-8 -*-

from __future__ import print_function, absolute_import, division

import uuid
import urllib
import logging

from .post_pusher import PostPusher
from ..utils.json import encode_json


class YaStatMartDistributionAlertsPusher(PostPusher):
    def __init__(self, config, do_push=True):
        super(YaStatMartDistributionAlertsPusher, self).__init__(do_push=do_push)
        self.logger = logging.getLogger(__name__)
        self.config = config

    def push(self, alerts):
        mart_url = '{}{}'.format(self.config['ya_stat_url'], self.config['mart_path'])
        return super(YaStatMartDistributionAlertsPusher, self).push(
            url=mart_url,
            json=self.__build_widgets_json(alerts),
            headers=self.config['ya_stat_auth']['auth_headers']
        )

    def __build_widgets_json(self, alerts):
        columns_count = 12
        col_span = 2
        widget_height = 280
        tab_id = uuid.uuid4()
        widget_description = {
            'tabs': [
                {
                    'id': format(tab_id),
                    'title': 'New tab',
                    'widgets': []
                }
            ]
        }
        for serial_number, alert in enumerate(alerts):
            self.logger.info('Sending alert: {}'.format(alert))
            row = int(col_span * serial_number / columns_count)
            column = int(col_span * serial_number - (row * columns_count))
            comment = tuple(
                value
                for dimension, value in alert['slice']
                if dimension in alert['dimensions'] and value != '_total_'
            )
            uri = self.__build_uri(alert['event'], alert['slice'], alert['date_then'], alert['date_now'])
            widget_id = uuid.uuid4()
            widget_description['tabs'][0]['widgets'].append(
                {
                    'colSpan': col_span,
                    'column': column,
                    'height': widget_height,
                    'id': format(widget_id),
                    'isTransparent': False,
                    'tab': 0,
                    'row': row,
                    'tabId': format(tab_id),
                    'type': 'chart',
                    'data': [
                        {
                            'chartLegend': 'default',
                            'chartSize': format(widget_height),
                            'chartTitle': 'default',
                            'commentContent': format(comment),
                            'commentMarkup': format(comment),
                            'content': format(comment),
                            'hideComments': False,
                            'href': uri,
                            'id': '{}'.format(uuid.uuid4()),
                            'title': alert['event'],
                        }
                    ]
                }
            )
        return widget_description

    def __build_uri(self, event, slice, date_min, date_max, scale='d'):
        uri = '{}{}'.format(self.config['ya_stat_url'], self.config['project_path'])
        uri += '?_fr2=1'
        graph_json_config = {'graphs': [{'fields': event, 'scale': scale, 'x_field': 'fielddate'}]}
        uri += '&_graph_json_config={}'.format(urllib.quote(encode_json(graph_json_config)))
        uri += '&_type=visual'
        for dimension, value in slice:
            uri += '&{}={}'.format(dimension, value)
        uri += '&visualizer=HighCharts'
        uri += '&_graph_title={}-{}'.format(date_min, date_max)
        uri += '&_graph_type=line'
        uri += '&date_min={}+00%3A00%3A00'.format(date_min)
        uri += '&date_max={}+00%3A00%3A00'.format(date_max)
        return uri
