import json
import requests
import xml.etree.ElementTree as ET
import util
import re

# -----------------------------------------------------------------------
# Parent API Class
# -----------------------------------------------------------------------

class __direct_api_parent:
    def __init__(self, token):
        self.token = token

    def meta_request(self, url, data=None, headers=None):
        if url.find('/v4/') >= 0:
            api_ver = 4
        elif url.find('/v5/') >= 0:
            api_ver = 5
        else:
            raise Exception('Unknown API version. URL is {0}'.format(url))

        response = requests.post(url, data=data, headers=headers)

        if response.status_code != 200:
            raise Exception('Error {0}: {1}'.format(res.status, res.text))

        try:
            if api_ver == 4:
                result = json.loads(response.text)['data']
            else:
                result = json.loads(response.text)['result']
        except KeyError:
            raise Exception('Data key is missing in {0}'.format(response.text))

        return result


# -----------------------------------------------------------------------
# V4 API Class
# -----------------------------------------------------------------------

class Version4(__direct_api_parent):
    def api_url(self):
        return 'https://api.direct.yandex.ru/v4/json/'

    def request(self, method, locale='ru', param=None):
        data = {
            'method': method,
            'token': self.token,
            'locale': locale,
        }

        if param != None:
            data['param'] = param

        jdata = json.dumps(data, ensure_ascii=False).encode('utf8')
        return self.meta_request(self.api_url(), data=jdata)

    def get_summary_stat(self, campaign_ids, start_date, end_date):
        param = {
            "CampaignIDS": campaign_ids,
            "StartDate": start_date,
            "EndDate": end_date
        }
        response = self.request("GetSummaryStat", param=param)
        return response

    def get_client_units(self, login):
        return self.request("GetClientsUnits", param=[login])

    def get_banners(self, campaign_ids):
        param = {
            "CampaignIDS": campaign_ids,
            "GetPhrases": 'No',
            "Filter": {
                "StatusArchive": ['No'],
            }
        }

        banners = []
        for i in range(0, len(campaign_ids), 10):
            campaigns = campaign_ids[:i + 10]
            response = self.request('GetBanners', param=param)
            banners += response
        return banners


# -----------------------------------------------------------------------
# V5 API Class
# -----------------------------------------------------------------------

class Version5(__direct_api_parent):
    def api_url(self, api_point=None):
        url = 'https://api.direct.yandex.ru/json/v5/'
        if api_point != None:
            url += api_point
        return url

    def request(self, method, login, params=None, api_point=None):
        headers = {
            'Authorization': "Bearer {0}".format(self.token),
            'Client-Login': login
        }

        data = {
            'method': method,
            'params': params
        }

        jdata = json.dumps(data, ensure_ascii=False).encode('utf8')
        return self.meta_request(self.api_url(api_point), headers=headers, data=jdata)


    def get_campaigns_list(self, login):
        params = {
            'SelectionCriteria': {
                'States': ['ON', 'OFF', 'ENDED', 'SUSPENDED'],
                'Statuses': ['ACCEPTED']
            },
            'FieldNames': ["Id", "Name", 'State'],

        }
        response = self.request("get", login, params=params, api_point='campaigns')
        try:
            result = response['Campaigns']
        except KeyError:
            raise Exception('Campaigns key is missing in {0}'.format(response.text))

        return result

    def get_banners(self, login, campaign_ids):
        params = {
            'SelectionCriteria': {
                'States': ['ON', 'OFF', 'SUSPENDED'],
                'Statuses': ['ACCEPTED'],
                'Types': ["TEXT_AD"],
                'CampaignIds': campaign_ids
            },
            'FieldNames': ["CampaignId"],
            'TextAdFieldNames': ["Href"],
            'MobileAppAdFieldNames': ["TrackingUrl"]

        }
        response = self.request("get", login, params=params, api_point='ads')
        try:
            result = response['Ads']
        except KeyError:
            raise Exception('Ads key is missing in {0}'.format(response))

        return result
