import datetime
import logging

import typing as tp
from flask import Flask, request, send_from_directory

from chyt import build_full_board_html, build_full_author_html, get_author_login_by_id
from config import LOCATION_SELECTORS
from exceptions import ValidationError
from libs import view_errors
from parse_board_selector import get_board_id_by_selector

app = Flask(__name__)
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)

# TODO: return some kind of special instance that carries all this info so that a change in the
#  output of this function need not be carried to the interface of other functions each time when a parameter is added
def parse_common_params(
    request_params: tp.Mapping[str, tp.Any]
) -> tp.Tuple[datetime.date, str, int, bool, tp.Tuple[str, ...]]:
    """

    :param request_params:
    :return:    end_date - the ending date for the graph
                freq - the timescale of the graph
                horizon - how many days before the end_date to draw
    """
    try:
        end_date = datetime.datetime.strptime(request_params.get('end_date', ''), '%Y-%m-%d').date()
    except ValueError:
        end_date = datetime.datetime.now().date()
    freq = request_params.get('freq', 'd')
    horizon = int(request_params.get('horizon', 30))
    split_ui = bool(request_params.get('split_ui', False))

    location_selector = request_params.get('location_selector', '')
    if not location_selector:
        location_selector = 'boards'
    if location_selector not in LOCATION_SELECTORS:
        raise ValidationError(
            f'''
            Указан неверный location_selector '{location_selector}', достпны: {LOCATION_SELECTORS}
            '''
        )

    possible_locations = tuple(loc for loc in LOCATION_SELECTORS[location_selector])

    if freq in ('min', 'T') and horizon > 1:
        raise ValidationError(
            '''Слишком большой горизонт для поминутной аггрегации. При поминутной аггрегации нельзя ставить горизонт
            больше дня (и меньше тоже :P). Необходимо добавить параметр horizon=1'''
        )
    return end_date, freq, horizon, split_ui, possible_locations


@app.route('/view_board')
@view_errors
def view_board_endpoint():
    the_params = dict(request.args)
    end_date, freq, horizon, split_ui, possible_locations = parse_common_params(the_params)
    board_id = get_board_id_by_selector(the_params['board_selector'])

    complete_html = build_full_board_html(
        board_id=board_id,
        end_date=end_date,
        freq=freq,
        horizon=horizon,
        split_ui=split_ui,
        possible_locations=possible_locations
    )
    return complete_html


@app.route('/view_author')
@view_errors
def view_author_endpoint():
    the_params = dict(request.args)
    end_date, freq, horizon, split_ui, possible_locations = parse_common_params(the_params)
    author_login = the_params.get('login', '')
    author_id = the_params.get('id', '')
    if not author_login and author_id:
        author_login = get_author_login_by_id(author_id)
    if not author_login:
        raise ValidationError('Не задан автор. Необходимо добавить параметр login=<ник автора в коллекциях>')
    return build_full_author_html(
        author_login,
        end_date=end_date,
        freq=freq,
        horizon=horizon,
        split_ui=split_ui,
        possible_locations=possible_locations
    )


@app.route('/css/<path:path>')
def send_css(path):
    return send_from_directory('css', path)


@app.route('/js/<path:path>')
def send_js(path):
    return send_from_directory('js', path)


if __name__ == '__main__':
    app.run(debug=True)
