import datetime
import logging
from abc import abstractmethod, ABC
from functools import wraps

import pandas as pd
import numpy as np
import requests
import typing as tp
import plotly.graph_objs as go
from pprint import pformat

from plotly.subplots import make_subplots

from config import INTERNAL_API_SECRET, INTERNAL_API_PUBLIC, PUBLIC_API, AVATARS_API
from exceptions import ViewableError, DataError

Json = tp.Mapping[str, tp.Any]
Card = Json
Board = Json
MarkerParameter = tp.Any

PLOTLY_SCRIPT = '<script src="js/plotly.v1.54.1.min.js"></script>'

LETTER_TEMPLATE = """
<html>
    <head>
    {plotly_script}
        <link rel="stylesheet" href="css/styles.css">
    </head>
    <body>
        <div><a href='{full_link}'>{full_link}</a></div>
        {board_summary_html}
    </body>
</html>
"""


def pformat_html(json: Json) -> str:
    return pformat(json).replace('\n', '<br/>')


def datetime_from_date(dt: datetime.date, hours: int = 0, minutes: int = 0, seconds: int = 0) -> datetime.datetime:
    return datetime.datetime(*dt.timetuple()[:3], hours, minutes, seconds)


def get_board_summary(board_id: str) -> tp.Mapping[str, tp.Any]:
    url = f'{INTERNAL_API_SECRET}/boards/{board_id}'
    response = requests.get(url)
    if response.status_code != 200:
        raise DataError(
            f'Api returned {response.status_code}, {response.text} when querying board {board_id} by url {url}'
        )
    return response.json()


def get_board_public_url(board: Board) -> str:
    return board.get('url', '').replace('internal.collections.yandex.ru', 'yandex.ru')


def get_board_cards(
    board_id: str,
    limit: tp.Optional[int] = None,
    sort: tp.Optional[str] = None,
) -> tp.Sequence[Card]:
    full_json = get_board_cards_meta(board_id, limit, sort)
    try:
        return full_json['results']
    except KeyError:
        logging.error(f'Could not find "results" key in {full_json}')
        raise


def get_board_cards_meta(
    board_id: str,
    page_size: tp.Optional[int] = None,
    sort: tp.Optional[str] = None,
    page: tp.Optional[int] = None
) -> Json:
    """
    :param page:
    :param board_id:
    :param page_size:
    :param sort: sorted
    :return:
    """
    params = {
        'page_size': page_size,
        'board.id': board_id,
        'sort': sort,
        'page': page
    }
    logging.debug(f'params are {params}')
    response = requests.get(f'{INTERNAL_API_PUBLIC}/cards', params)
    logging.debug(f'{response.status_code}')
    logging.debug(f'{response.text}')
    return response.json()


def get_author_cards_meta(
    login: str,
    page_size: tp.Optional[int] = None,
    sort: tp.Optional[str] = None,
    page: tp.Optional[int] = None
) -> Json:
    """
    :param page:
    :param login:
    :param page_size:
    :param sort: sorted
    :return:
    """
    params = {
        'page_size': page_size,
        'owner.login': login,
        'sort': sort,
        'page': page
    }
    logging.debug(f'params are {params}')
    response = requests.get(f'{INTERNAL_API_PUBLIC}/cards', params)
    logging.debug(f'{response.status_code}')
    logging.debug(f'{response.text}')
    return response.json()


def get_author_boards_meta(
    login: str,
    page_size: int = 100,
    sort: str = '-service.updated_at',
    page: int = 1
) -> tp.Mapping[str, tp.Any]:
    params = {
        'sort': sort,
        'saveable_for_user': login,
        'page_size': page_size,
        'page': page
    }
    url = f'{INTERNAL_API_SECRET}/boards'
    resp = requests.get(url, params)
    return resp.json()


def create_time_stopper(stop_at: datetime.datetime, last: bool = True) -> tp.Callable[[tp.Any], bool]:

    def time_is_valid(portion: tp.Any) -> bool:
        index = -1 if last else 0
        if not portion['results']:
            return False
        return parse_time(portion['results'][index]['service']['created_at']) < stop_at

    return time_is_valid


def gather_results_from_api(
    results_function: tp.Callable[[tp.Any], tp.Mapping[str, tp.Any]],
    params: tp.Dict[str, tp.Any],
    start_page: int = 1,
    max_pages: int = 100,
    page_size: int = 100,
    *,
    time_to_stop: tp.Optional[tp.Callable[[Json], bool]] = None
) -> tp.List[tp.Any]:
    """
    :param results_function:
    :param params:
    :param start_page: one-base start page!
    :param max_pages:
    :param page_size:
    :param time_to_stop: a predicate that should check the next portion of data and return True if there is no
    point to continue requests
    :return:
    """
    res = []
    params['page'] = start_page
    params['page_size'] = page_size
    for _ in range(max_pages):
        page_results = results_function(**params)
        res += list(page_results['results'])
        logging.debug(f"For page {start_page + _} there was {page_results['count']} results")
        if page_results['next'] is None or (time_to_stop is not None and time_to_stop(page_results)):
            break
        params['page'] = page_results['next']['page']
    return res


class GraphEventSeries(ABC):

    @abstractmethod
    def get_legend_name(self) -> str:
        pass

    @abstractmethod
    def get_index_series(self) -> pd.Series:
        """
        A datetime series
        :return:
        """
        pass

    @abstractmethod
    def get_values_series(self) -> pd.Series:
        pass

    def get_point_labels(self) -> tp.Optional[tp.Sequence[str]]:
        return None

    def get_marker_properties(self) -> tp.Mapping[str, MarkerParameter]:
        return dict(
            size=20,
            line=dict(width=2)
        )


class IndexOnlyEventSeries(GraphEventSeries, ABC):
    def __init__(self, index: pd.Series):
        self.index = index

    def get_values_series(self) -> pd.Series:
        return pd.Series(np.ones(len(self.index)))

    def get_index_series(self) -> pd.Series:
        return self.index


class CardCreateSeries(IndexOnlyEventSeries):
    def get_marker_properties(self) -> tp.Mapping[str, MarkerParameter]:
        return dict(
            color='LightSkyBlue',
            size=20,
            symbol=3,
            line=dict(
                color='MediumPurple',
                width=2
            )
        )

    def get_legend_name(self) -> str:
        return 'Создание карточки'


class CorgieEventSeries(IndexOnlyEventSeries):
    def get_marker_properties(self) -> tp.Mapping[str, MarkerParameter]:
        return dict(
            color='Green',
            size=20,
            symbol=4,
            line=dict(
                color='MediumPurple',
                width=2
            )
        )

    def get_legend_name(self) -> str:
        return 'Corgie проверка'


class BoardCreateSeries(IndexOnlyEventSeries):
    def get_legend_name(self) -> str:
        return 'Создание коллекции'

    def get_marker_properties(self) -> tp.Mapping[str, MarkerParameter]:
        return dict(
            color='Red',
            size=20,
            symbol=3,
            line=dict(
                color='MediumPurple',
                width=2
            )
        )


class PushSeries(IndexOnlyEventSeries):
    def __init__(self, index: pd.Series, push_type: str):
        super().__init__(index)
        self.push_type = push_type

    def get_legend_name(self) -> str:
        return f'Push: {self.push_type}'

    def get_marker_properties(self) -> tp.Mapping[str, MarkerParameter]:
        return dict(
            size=20,
            line=dict(
                color='Blue',
                width=2
            )
        )


def get_board_cards_create_events(
    board_id: str,
    max_pages: int = 100,
    page_size: int = 100,
    sort: str = '-service.created_at'
) -> CardCreateSeries:
    """
    :param sort:
    :param board_id:
    :param max_pages:
    :param page_size: 100 is maximum!
    :return:
    """
    results = gather_results_from_api(
        get_board_cards_meta,
        {'board_id': board_id, 'sort': sort},
        max_pages=max_pages,
        page_size=page_size
    )
    dttm_series = pd.Series([parse_time(card['service']['created_at']) for card in results])
    return CardCreateSeries(index=dttm_series)


def get_corgie_events(
    board: Board
) -> tp.Sequence[CorgieEventSeries]:
    res = []
    times = []
    feats = board.get('features', {})
    try:
        times.append(datetime.datetime.utcfromtimestamp(feats['corgie_v2']['ts'] / 1000))
    except KeyError:
        pass
    try:
        times.append(datetime.datetime.utcfromtimestamp(feats['corgie_v3']['ts'] / 1000))
    except KeyError:
        pass
    if times:
        res.append(CorgieEventSeries(pd.Series(times)))
    return res


def get_corgieness(board: Board) -> tp.Dict[str, str]:
    return board.get('features', {
        'corgie_v2': 'no_info',
        'corgie_v3': 'no_info'
    })


def get_board_summary_html(
    board: Board,
    cards: tp.Iterable[Card] = (),
    under_description_html: str = '',
    additional_stats: tp.Optional[Json] = None
) -> str:
    url = get_board_public_url(board)
    stats_str = pformat_html(board['stat'])
    if additional_stats is not None:
        additional_stats = dict(additional_stats)
        features = get_corgieness(board)
        additional_stats.update(features)
    else:
        additional_stats = {'features': board['features']}
    additional_stats_str = pformat_html(additional_stats)
    author_viewer_url = ''
    header = f"""
    <h1>{board['title']}</h1>
    <div><a href='{url}'><img src='{get_board_teaser_url(board)}' alt='{board['title']}' style='width: 400px'></a></div>
    <div><a href='{url}'>{url}</a></div>
    <div><a href='{url}'>{board['id']}</a></div>
    <div><a href='view_author?login={board['owner']['login']}' target='_blank'>Показать автора</a></div>
    <div style='width:45%; display:inline-block; vertical-align: top;'>
        <h2>Online stats</h2><div>{stats_str}</div>
    </div>
    <div style='width:45%; display:inline-block; vertical-align: top;'>
        <h2>Additional stats</h2><div>{additional_stats_str}</div>
    </div>
    <div>{board.get('document', {}).get('description', '')}</div>
    {under_description_html}
    """
    body = ''
    for card in cards:
        body += get_card_block_html(card)
    return header + body


def parse_time(date_str: str) -> datetime.datetime:
    return datetime.datetime.strptime(date_str[:19], '%Y-%m-%dT%H:%M:%S')


def reformat_time(date_str: str) -> str:
    return format_time(parse_time(date_str))


def format_time(dttm: datetime.datetime) -> str:
    return dttm.strftime('%Y-%m-%d %H:%M:%S')


def get_card_block_html(card: Card) -> str:
    return f"""
    <div class='card'>
    <div style='height: 250px'>
        <img src='{get_card_img_url(card)}' style='max-width:250px;max-height:250px; margin: auto; display: block;'/>
    </div>
    <p>Created: {reformat_time(card['service']['created_at'])},
    Updated: {reformat_time(card['service']['updated_at'])}</p>
    <p>{card['description']}</p>
    <div><a href='{PUBLIC_API}/card/{card['id']}'>{card['id']}</a></div>
    <br/>
    </div>
    """


def get_board_block_html(board: Board) -> str:
    url = get_board_public_url(board)
    selected_stats = get_corgieness(board)
    SELECTED_KEYS = ('cards_count', 'shares_count', 'subscribers_count', 'views_count')
    stats = board['stat']
    for key in SELECTED_KEYS:
        if stats.get(key, 0):
            selected_stats[key] = stats[key]
    return f"""
    <div class='card'>
    <h3 class='card-header'>{board.get('title', 'no title')}</h3>
    <div style='height: 250px'>
        <img src='{get_board_teaser_url(board)}' style='max-width:250px;max-height:250px; margin: auto; display: block;'/>
    </div>
    <p>Created: {reformat_time(board['service']['created_at'])},
    Updated: {reformat_time(board['service']['updated_at'])}</p>
    <p>{pformat_html(selected_stats)}</p>
    <p>{board.get('description', '')}</p>
    <div><a href='{url}'>{board['title']}</a></div>
    <div><a class href='view_board?board_selector={board['id']}' target='_blank'>viewer</a></div>
    <br/>
    </div>
    """


def get_card_img_url(card: Card) -> str:
    content = card['content'][0]['content']
    return get_avatar_url(content['group_id'], content['avatars_key'])


def get_board_teaser_url(board: Board) -> str:
    teaser_info = board['teaser_info_v2']
    return get_avatar_url(teaser_info['group_id'], teaser_info['avatars_key'], is_teaser=True)


def get_avatar_url(group_id: str, avatars_key: str, size: str = 'preview', is_teaser: bool = False, **kwargs) -> str:
    """
    :param is_teaser:
    :param group_id:
    :param avatars_key:
    :param size: preview/orig/ some size
    :return:
    """
    teaser_mixin = '-teasers' if is_teaser else ''
    return f'{AVATARS_API}/get-pdb{teaser_mixin}/{group_id}/{avatars_key}/{size}'


def convert_to_numeric(df: pd.DataFrame, column: str) -> None:
    """
    Mutates the initial dataframe!
    :param df:
    :param column:
    :return:
    """
    df[column] = pd.to_numeric(df[column])


def convert_to_datetime(df: pd.DataFrame, column: str, timezone_offset: int = 3) -> None:
    """
    Mutates the initial dataframe!
    :param df:
    :param column:
    :param timezone_offset:
    :return:
    """
    convert_to_numeric(df, column)
    # TODO: handle timezones in more accurate way then adding offset in seconds
    df[column] = pd.to_datetime((df[column] + 3600 * timezone_offset) * 1000000000)


def get_shows_clicks_by_json(parsed: tp.List[Json]) -> pd.DataFrame:
    if not len(parsed):
        return pd.DataFrame()
    shows_clicks = pd.DataFrame.from_records(parsed)
    convert_to_numeric(shows_clicks, 'shows')
    convert_to_numeric(shows_clicks, 'clicks')
    convert_to_datetime(shows_clicks, 'timestamp')
    shows_clicks['datetime'] = shows_clicks['timestamp'].rename('datetime')
    return shows_clicks


def get_pushes_by_json(parsed: tp.Sequence[Json]) -> pd.DataFrame:
    if not len(parsed):
        return pd.DataFrame()

    pushes = pd.DataFrame.from_records(parsed)
    convert_to_numeric(pushes, 'timestamp')
    convert_to_datetime(pushes, 'timestamp')
    pushes['datetime'] = pushes['timestamp'].rename('datetime')
    return pushes


def get_push_events_from_dataframe(pushes: pd.DataFrame) -> tp.Sequence[PushSeries]:
    aggregated = pushes.groupby(['push_type', 'datetime']).count()
    serieses = []
    for push_type in pd.unique(aggregated.index.get_level_values(0)):
        serieses.append(PushSeries(pd.Series(aggregated.loc[push_type].index), push_type))
    return serieses


def aggregate_and_fill_values(
    shows_clicks: pd.DataFrame,
    start_datetime: datetime.datetime,
    end_datetime: datetime.datetime,
    freq: str
) -> tp.Mapping[str, pd.DataFrame]:
    if len(shows_clicks):
        rounded = shows_clicks['datetime'].dt.floor(freq)
        aggregated = shows_clicks.groupby([shows_clicks['groupper'], rounded]).agg({
            'shows': ['sum'],
            'clicks': ['sum']
        })
        aggregated.columns = ['shows', 'clicks']
    else:
        aggregated = pd.DataFrame(columns=['shows', 'clicks', 'datetime'])
    resulting_dfs = {}
    all_times = pd.date_range(start_datetime, end_datetime, freq=freq).floor(freq)
    for location in aggregated.index.get_level_values(0):
        df = aggregated.loc[location]
        df = df.reindex(all_times, fill_value=0).reset_index()
        df.rename(columns={'index': 'date'}, inplace=True)
        resulting_dfs[location] = df
    return resulting_dfs


def plotly_draw_area(
    aggregated: tp.Mapping[str, pd.DataFrame],
    *,
    fig: tp.Optional[go.Figure] = None, secondary: bool = False, y_key: str = 'shows', x_key: str = 'date'
) -> go.Figure:
    if fig is None:
        fig = go.Figure()
        secondary = False
    items = list(sorted(list(aggregated.items()), key=lambda itm: itm[1]['shows'].sum()))
    for location, df in items:
        fig.add_trace(go.Scatter(
                x=df[x_key], y=df[y_key],
                hoverinfo='x+y',
                mode='lines',
                name=location,
                stackgroup=y_key,
                hovertemplate=y_key + ': %{y} date: %{x} location_ui: ' + location,
            ),
            secondary_y=secondary
        )
    return fig


def plotly_draw_events(
    events: GraphEventSeries, freq: str = 'd', start_datetime: tp.Optional[datetime.datetime] = None,
    *,
    fig: tp.Optional[go.Figure] = None, secondary: bool = False
) -> go.Figure:
    """
    Draws a single series on the graph
    :param events:
    :param freq:
    :param start_datetime:
    :param fig:
    :param secondary:
    :return:
    """
    if fig is None:
        fig = go.Figure()
        secondary = False
    if not events:
        return fig
    events_df = pd.DataFrame.from_dict(data={
        'summand': events.get_values_series(),
        'dttm': events.get_index_series()
    })
    if start_datetime is not None:
        events_df = events_df[events_df['dttm'] > start_datetime]
    if not len(events_df):
        return fig
    res = events_df.groupby(events_df['dttm'].dt.floor(freq))['summand'].sum()
    labels = events.get_point_labels()
    fig.add_trace(go.Scatter(x=res.index, y=res,
                             mode='markers',
                             name=events.get_legend_name(),
                             text=labels,
                             marker=events.get_marker_properties(),
                             ),
                  secondary_y=secondary)
    return fig


def add_heading(fig: go.Figure, heading: str) -> go.Figure:
    fig.update_layout(
        title={
            'text': heading,
            'x': 0.5,
            'xanchor': 'center',
            'yanchor': 'top',
            'font': {
                'family': 'Courier New, monospace',
                'size': 26,
                'color': 'black'
            }
        }
    )
    return fig


def draw_plotly_area_and_events(
    aggregated_dfs: tp.Mapping[str, pd.DataFrame],
    event_serieses: tp.Sequence[GraphEventSeries],
    start_datetime: datetime.datetime,
    y_key: str = 'shows',
    heading: str = 'Показы и добавления карточек',
    freq: str = 'd'
) -> go.Figure:
    fig = make_subplots(specs=[[{"secondary_y": True}]])
    fig = plotly_draw_area(aggregated_dfs, y_key=y_key, fig=fig)
    for event_series in event_serieses:
        fig = plotly_draw_events(event_series, freq=freq, start_datetime=start_datetime, fig=fig, secondary=True)
    add_heading(fig, heading)
    return fig


def get_html_link() -> str:
    try:
        import nirvana.job_context as nv
    except ImportError:
        return ''
    ctx = nv.context()
    outputs = ctx.get_outputs()
    item = outputs.get_item('html_report')
    return item.get_download_url()


def main(in1, in2, in3, mr_tables, token1=None, token2=None, param1=None, param2=None, html_file=None):
    # obtain parameters
    input_params = in2[0]
    board_id = input_params['$board_id']
    current_datetime = datetime.datetime.strptime(input_params['$date'], '%Y-%m-%d')

    nirvana_link = get_html_link().replace('/data', '/content')

    # some fixed constants, should be moved to parameters may be
    horizon = 30  # how many days before $date should be drawn
    freq = 'd'  # the level of aggregation populated to pandas reindex

    additional_stats = in3[0]

    complete_html = create_board_html(board_id, current_datetime, in1, additional_stats, horizon, freq, nirvana_link)

    # write output
    html_file.write(complete_html)


def create_graphs_html(
    aggregated_dfs: tp.Mapping[str, pd.DataFrame],
    event_serieses: tp.Sequence[GraphEventSeries],
    start_datetime: datetime.datetime,
    freq: str = 'd'
) -> str:
    shows_fig = draw_plotly_area_and_events(
        aggregated_dfs, event_serieses, start_datetime, 'shows', 'Показы и добавления карточек', freq
    )
    clicks_fig = draw_plotly_area_and_events(
        aggregated_dfs, event_serieses, start_datetime, 'clicks', 'Клики и добавления карточек', freq
    )

    shows_html = shows_fig.to_html(full_html=False, include_plotlyjs=False)
    clicks_html = clicks_fig.to_html(full_html=False, include_plotlyjs=False)
    return f'<div>{shows_html}</div><div>{clicks_html}</div>'


def create_board_html(
    board_id: str,
    filter_datetime: datetime.datetime,
    shows_clicks_json: tp.List[Json],
    additional_stats: Json,
    horizon: int = 30,
    freq: str = 'd',
    nirvana_link: str = ''
) -> str:
    """

    :param board_id:
    :param filter_datetime:
    :param shows_clicks_json: the data on shows and clicks
    :param additional_stats: custom data to be shown in additional stats as a prettified json
    :param horizon: how many days before filter_datetime should be drawn
    :param freq: the level of aggregation populated to pandas reindex
    :param nirvana_link:
    :return:
    """

    # TODO: make this function for pure display. Move getting the data outside
    # prepare raw data
    board = get_board_summary(board_id)
    shows_clicks = get_shows_clicks_by_json(shows_clicks_json)
    events = [get_board_cards_create_events(board_id, max_pages=5, page_size=100)]
    start_datetime = filter_datetime - datetime.timedelta(days=horizon)

    events += get_corgie_events(board)

    # process data
    aggregated_dfs = aggregate_and_fill_values(shows_clicks, start_datetime, filter_datetime, freq)

    # add service stats
    additional_stats_extended = {k: v for k, v in additional_stats.items()}
    additional_stats_extended.update(board['service'])

    # prepare htmls
    graphs_html = create_graphs_html(aggregated_dfs, events, start_datetime, freq)
    board_summary_html = get_board_summary_html(
        board,
        get_board_cards(board_id, 20, sort='-service.created_at'),
        graphs_html,
        additional_stats_extended
    )

    # unite htmls
    complete_html = LETTER_TEMPLATE.format(
        plotly_script=PLOTLY_SCRIPT,
        full_link=nirvana_link,
        board_summary_html=board_summary_html
    )
    return complete_html


def view_errors(endpoint):
    @wraps(endpoint)
    def wrapper():
        try:
            return endpoint()
        except ViewableError as e:
            return f'''
            <html>
                <body>
                    <h1>{e.__class__.__name__}</h1>
                    <div>{e.get_error()}</div>
                </body>
            </html>
            '''

    return wrapper
