#!/usr/bin/env python
# coding=utf-8

import requests
import json
import statface_client
import argparse
import sys

from time import sleep


def parse_args():
    parser = argparse.ArgumentParser(
        description="Upload km queue statistics",
        formatter_class=argparse.ArgumentDefaultsHelpFormatter
    )

    parser.add_argument("-t", "--token", required=True, help="Oauth token")

    return parser.parse_args()


def get_report(oauth_token):
    client = statface_client.StatfaceClient(
        host='upload.stat.yandex-team.ru',
        oauth_token=oauth_token
    )

    report = client.get_report('Collections/Metrics/km/podbmc_ticket_changelogs')

    return report


def main():
    args = parse_args()

    page_number = 1
    oauth_token = args.token

    issues_status_change_log = {}
    issues_processed = 0
    while True:
        try:
            issues_request = requests.get(
                'https://st-api.yandex-team.ru/v2/issues?filter=queue:PODBCM&page={}&'.format(page_number),
                headers={'Authorization': oauth_token}
            )
        except ConnectionError:
            sleep(360)
            continue

        if issues_request.status_code != 200:
            break

        issues_json = json.loads(issues_request.text)
        for issue in issues_json:
            if 'key' not in issue:
                continue

            try:
                change_log_request = requests.get(
                    'https://st-api.yandex-team.ru/v2/issues/{}/changelog'.format(issue['key']),
                    headers={'Authorization': oauth_token}
                )
            except ConnectionError:
                sleep(360)
                continue

            if change_log_request.status_code != 200:
                break

            change_log = json.loads(change_log_request.text)

            first_change = True
            for change in sorted(change_log, key=lambda x: x['updatedAt']):
                change_data = {}
                change_data['change_date'] = change['updatedAt'][:10]
                status_change = False
                for field in change.get('fields', []):
                    if field.get('field', {}).get('id') == 'status':
                        status_change = True
                        if isinstance(field.get('from'), dict):
                            if first_change:
                                change_data['old_status'] = ''
                                first_change = False
                            else:
                                change_data['old_status'] = field['from'].get('key', '')
                        else:
                            change_data['old_status'] = ''

                        if isinstance(field.get('to'), dict):
                            change_data['new_status'] = field['to'].get('key', '')
                        else:
                            change_data['new_status'] = ''

                    if field.get('field', {}).get('id') == 'resolution':
                        if isinstance(field.get('to'), dict):
                            change_data['new_status'] += '_' + field['to'].get('key', '')
                        else:
                            change_data['new_status'] += '_ '

                if status_change:
                    if change_data['change_date'] in issues_status_change_log:
                        if change_data['old_status'] in issues_status_change_log[change_data['change_date']]:
                            issues_status_change_log[change_data['change_date']][change_data['old_status']] -= 1
                        else:
                            issues_status_change_log[change_data['change_date']][change_data['old_status']] = -1

                        if change_data['new_status'] in issues_status_change_log[change_data['change_date']]:
                            issues_status_change_log[change_data['change_date']][change_data['new_status']] += 1
                        else:
                            issues_status_change_log[change_data['change_date']][change_data['new_status']] = 1
                    else:
                        issues_status_change_log[change_data['change_date']] = {}
                        if change_data['old_status'] in issues_status_change_log[change_data['change_date']]:
                            issues_status_change_log[change_data['change_date']][change_data['old_status']] -= 1
                        else:
                            issues_status_change_log[change_data['change_date']][change_data['old_status']] = -1

                        if change_data['new_status'] in issues_status_change_log[change_data['change_date']]:
                            issues_status_change_log[change_data['change_date']][change_data['new_status']] += 1
                        else:
                            issues_status_change_log[change_data['change_date']][change_data['new_status']] = 1

            issues_processed += 1
            sleep(1)

            sys.stderr.write('issues_processed {}'.format(issues_processed))

            sys.stderr.write('page_number {}'.format(page_number))
        page_number += 1

    status_map = {
        '': 'empty',
        'new': 'new',
        'open': 'open',
        'inProgress': 'in_progress',
        'proposalSent': 'proposal_sent',
        'closed_refusal': 'closed_refusal',
        'instructionsSent': 'instructions_sent',
        'closed_inappropriate': 'closed_innaproriate',
        'support': 'support',
        'registration': 'registration',
        'negotiations': 'negotiations',
        'readyForVerifying': 'ready_for_verifying',
        'closed_noReply': 'closed_no_reply',
        'accepting': 'accepting',
        'closed_workStopped': 'closed_work_stopped',
        'noContactInformation': 'no_contact_information',
        'closed': 'closed',
        'friendsRequested': 'friends_requested',
        'closed_inappropriateaAfterProposal': 'closed_inappropriatea_after_proposal',
        'closed_duplicate': 'closed_duplicate',
        'paused': 'paused',
        'unableToContact': 'unable_to_contact'
    }

    statface_dict = {x: 0 for x in status_map.values()}
    statface_dict['fielddate'] = ''

    statface_data = []
    for change_date in sorted(issues_status_change_log.keys()):
        for status in issues_status_change_log[change_date]:
            statface_dict['fielddate'] = change_date
            statface_dict[status_map[status]] += issues_status_change_log[change_date][status]

        statface_data.append(statface_dict.copy())

    report = get_report(oauth_token)
    report.upload_data(scale='daily', data=statface_data)


if __name__ == "__main__":
    main()
