# coding: utf-8

from collections import Counter
from datetime import timedelta

from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from analytics.plotter_lib.utils import date_range, get_dts_delta, AddTotalsMapper
from analytics.collections.plotter_collections.plots.utils import format_client_name, format_client_ui
from pdb_analytics.corgie import get_slug_verdict

from nile.api.v1 import (
    extended_schema,
    extractors as ne,
    aggregators as na,
    filters as nf,
    with_hints,
    Record,
)
from qb2.api.v1 import (
    typing as qt,
    filters as qf
)


class BaseActiveUsersReducer(object):
    def __init__(self, action_type, calc_only_days=False, date=None, dateend=None):
        self.action_type = action_type
        self.calc_only_days = calc_only_days
        self.date = date
        self.dateend = dateend

    def date_iterator(self, date, act_dates, user_type, period, min_actions_count, record_key):
        raise NotImplementedError('date_iterator not implemented (see {})'.format(self.__class__))

    def __call__(self, groups):
        """
        :param groups: Набор туплов: характеристики юзера и список редир записей
        :return: обновленные записи, для каждого юзер-дня подсчитано, вернулся ли юзер в течение
        следующих period дней (period перебирается)
        """
        for key, records in groups:
            user_type = key.user_type
            # счетчик сколько раз в конкретный день выполнено целевое действие
            # предпосчет, чтобы потом пройтись циклом по дням
            act_dates = Counter()
            for r in records:
                act_dates[r.fielddate] += 1

            for period in self.PERIODS:
                for min_actions_count in self.ACTIONS_COUNT:
                    for date in date_range(
                        self.date - timedelta(days=period),
                        self.dateend + timedelta(days=period),
                    ):
                        r = self.date_iterator(
                            date,
                            act_dates,
                            user_type=user_type,
                            period=period,
                            min_actions_count=min_actions_count,
                            record_key=key
                        )
                        if r:
                            yield r


class BracketRetentionReducer(BaseActiveUsersReducer):
    def __init__(self, *args, **kwargs):
        self.PERIODS = (1, 7, 14, 30)
        self.ACTIONS_COUNT = (1, 2)
        self.key_fields = []
        if 'key_fields' in kwargs:
            self.key_fields = kwargs['key_fields']
            del kwargs['key_fields']
        super(BracketRetentionReducer, self).__init__(*args, **kwargs)

    def date_iterator(self, date, act_dates, user_type, period, min_actions_count, record_key):
        date_str = date.strftime(DATE_FORMAT)
        if date_str not in act_dates:
            # считаем только даты, за которых был визит или целевое событие
            return

        current_actions_count = 0
        for date_iter in date_range(date + timedelta(days=1), date + timedelta(days=period)):
            # для каждой даты итерируемся на `period` дней вперёд и считаем actions_count за этот период
            # `period` не включает в себя текущий день
            value = act_dates.get(date_iter.strftime(DATE_FORMAT), 0)
            if value > 0:
                current_actions_count += 1 if self.calc_only_days else value

        outrec = dict(
            fielddate=date_str,
            user_type=user_type,
            action_type=self.action_type,
            period=str(period),
            actions_count=str(min_actions_count),
            returned=1 if current_actions_count >= min_actions_count else 0
        )
        for key_field in self.key_fields:
            outrec[key_field] = record_key.get(key_field)
        return Record.from_dict(outrec)


class ActiveUsersReducer(BaseActiveUsersReducer):
    def __init__(self, *args, **kwargs):
        self.PERIODS = (1, 7, 14, 30)
        self.ACTIONS_COUNT = (1, 2, 3, 5)

        self.key_fields = []
        if 'key_fields' in kwargs:
            self.key_fields = kwargs['key_fields']
            del kwargs['key_fields']
        super(ActiveUsersReducer, self).__init__(*args, **kwargs)

    def date_iterator(self, date, act_dates, user_type, period, min_actions_count, record_key):
        if date > self.dateend or date < self.date:
            # считаем только за указанный период
            return

        current_actions_count = 0
        for date_iter in date_range(date - timedelta(days=period-1), date):
            # для каждой даты итерируемся на `period` дней назад и считаем actions_count за этот период
            # `period` включает в себя текущий день `date`, который последний
            value = act_dates.get(date_iter.strftime(DATE_FORMAT), 0)
            if value > 0:
                current_actions_count += 1 if self.calc_only_days else value

        if current_actions_count >= min_actions_count:
            outrec = dict(
                fielddate=date.strftime(DATE_FORMAT),
                user_type=user_type,
                action_type=self.action_type,
                period=str(period),
                actions_count=str(min_actions_count),
                actions=current_actions_count,
            )
            for key_field in self.key_fields:
                outrec[key_field] = record_key.get(key_field)
            return Record.from_dict(outrec)


class ActiveUsers(Plot):
    @require('PrepBoards.with_organic_flag', 'Corgie.corgie', 'PrepCards.board_first_cards')
    def get_created_boards(self, streams):
        date_str = self.date.strftime(DATE_FORMAT)
        boards = streams['PrepBoards.with_organic_flag'].project(
            'board_id',
            'board_slug',
            'user_type',
            user_id='board_owner_id',
            is_private='board_is_private',
            fielddate='board_created_at_date',
            is_corgie_v2=ne.custom(lambda x: True if x is not None and x == 'ok' else False, 'board_features_corgie_v2').with_type(bool),
            is_corgie_v3=ne.custom(lambda x: True if x is not None and x in ('2', '3') else False, 'board_features_corgie_v3').with_type(bool),
        )
        corgie = streams['Corgie.corgie'].project(
            'board_id',
            is_corgie_v1=ne.const(True).with_type(bool),
        )
        cards = streams['PrepCards.board_first_cards'].project(
            board_id='card_board_id',
            ui='card_origin_ui',
            origin_client=ne.custom(format_client_name, 'card_origin_client_name').with_type(str),
            origin_ui=ne.custom(format_client_ui, 'card_origin_client_ui').with_type(str),
            origin_action=ne.custom(lambda x: x if x else 'empty', 'card_origin_action').with_type(str)
        )

        return boards \
            .join(cards, type='inner', by='board_id') \
            .join(corgie, type='left', by='board_id') \
            .filter(
                nf.and_(
                    nf.custom(lambda x: get_slug_verdict(x) == 'good', 'board_slug'),
                    nf.custom(lambda date: get_dts_delta(date_str, date).days <= 60, 'fielddate')
                )
            ) \
            .project(
                'user_id',
                'user_type',
                'board_id',
                'ui',
                'origin_client',
                'origin_ui',
                'origin_action',
                'is_private',
                'fielddate',
                is_corgie_v1=ne.custom(lambda x: False if x is None else x, 'is_corgie_v1').with_type(bool),
                is_corgie_v2=ne.custom(lambda x: False if x is None else x, 'is_corgie_v2').with_type(bool),
                is_corgie_v3=ne.custom(lambda x: False if x is None else x, 'is_corgie_v3').with_type(bool),
            )

    @require('PrepCards.with_organic_flag', 'Corgie.corgie')
    def get_created_cards(self, streams):
        date_str = self.date.strftime(DATE_FORMAT)
        cards = streams['PrepCards.with_organic_flag'].project(
            'card_id',
            'user_type',
            'card_board_id',
            user_id='card_owner_id',
            ui='card_origin_ui',
            origin_client=ne.custom(format_client_name, 'card_origin_client_name').with_type(str),
            origin_ui=ne.custom(format_client_ui, 'card_origin_client_ui').with_type(str),
            origin_action=ne.custom(lambda x: x if x else 'empty', 'card_origin_action').with_type(str),
            is_private='card_is_private',
            fielddate='card_created_at_date',
        )
        corgie = streams['Corgie.corgie'].project(
            'board_id',
            is_corgie_v1=ne.const(True).with_type(bool),
            is_corgie_v2=ne.custom(lambda x: True if x is not None and x == 'ok' else False, 'board_features_corgie_v2').with_type(bool),
            is_corgie_v3=ne.custom(lambda x: True if x is not None and x in ('2', '3') else False, 'board_features_corgie_v3').with_type(bool),
        )

        return cards \
            .join(corgie, type='left', by_left='card_board_id', by_right='board_id') \
            .filter(nf.custom(lambda date: get_dts_delta(date_str, date).days <= 60, 'fielddate')) \
            .project(
                'user_id',
                'user_type',
                'ui',
                'origin_client',
                'origin_ui',
                'origin_action',
                'is_private',
                'fielddate',
                is_corgie_v1=ne.custom(lambda x: False if x is None else x, 'is_corgie_v1').with_type(bool),
                is_corgie_v2=ne.custom(lambda x: False if x is None else x, 'is_corgie_v2').with_type(bool),
                is_corgie_v3=ne.custom(lambda x: False if x is None else x, 'is_corgie_v3').with_type(bool),
            )

    @require('CollectionsRedirLog.full_with_additional_days')
    def get_own_cards_accesses(self, streams):
        return streams['CollectionsRedirLog.full_with_additional_days'] \
            .filter(nf.and_(
                qf.nonzero('card_id'),
                nf.equals('path', 'access')),
                nf.equals('is_owner', 'true')
            ) \
            .project('fielddate', 'ui', 'user_id', 'yandexuid', 'puid')

    @require('CollectionsRedirLog.full_with_additional_days')
    def get_own_boards_accesses(self, streams):
        return streams['CollectionsRedirLog.full_with_additional_days'] \
            .filter(nf.and_(
                qf.nonzero('board_id'),
                nf.equals('path', 'access'),
                nf.equals('is_owner', 'true')
            )) \
            .project('fielddate', 'ui', 'user_id', 'yandexuid', 'puid')

    @require('CollectionsRedirLog.full_with_additional_days')
    def get_visits(self, streams):
        return streams['CollectionsRedirLog.full_with_additional_days'] \
            .filter(nf.equals('path', 'start.session')) \
            .project('fielddate', 'ui', 'user_id', 'yandexuid', 'puid')

    def calc_active_users(self, stream, action_type, calc_only_days=False):
        """Универсальный код вычисления активных пользователей
        Заданным окном `назад` считает количество людей, которые совершили N действий за период K дней"""
        KEY_FIELDS = ('fielddate', 'user_type', 'ui', 'action_type', 'actions_count', 'period')

        return stream \
            .map(with_hints(output_schema=extended_schema())(
                AddTotalsMapper(['user_type', 'ui'], ['fielddate', 'user_id'])
            )) \
            .groupby('user_id', 'user_type', 'ui') \
            .reduce(
                with_hints(output_schema=dict(
                    fielddate=qt.String,
                    user_type=qt.String,
                    ui=qt.String,
                    period=qt.String,
                    actions_count=qt.String,
                    action_type=qt.String,
                    actions=qt.UInt64,
                ))(
                    ActiveUsersReducer(action_type, calc_only_days, date=self.date, dateend=self.dateend, key_fields=['ui'])
                ),
                intensity=500,  # data_size_per_job ~= 2-3Mb
            ) \
            .groupby(*KEY_FIELDS) \
            .aggregate(
                users=na.count(),
                actions=na.sum('actions'),
            ) \
            .publish(self.get_statface_report('Collections/Metrics/authors/ActiveUsersV4'), allow_change_job=True)

    def calc_bracket_retention(self, stream, action_type, calc_only_days=False):
        """Универсальный код вычисления диапазонного ретеншна
        Заданным окном `вперёд` считает количество людей, которые совершили N действий за период K дней и долю вернувшихся"""
        KEY_FIELDS = ('fielddate', 'user_type', 'ui', 'action_type', 'actions_count', 'period')

        output = stream \
            .map(with_hints(output_schema=extended_schema())(
                AddTotalsMapper(['user_type', 'ui'], ['fielddate', 'user_id'])
            )) \
            .groupby('user_id', 'user_type', 'ui') \
            .reduce(
                with_hints(output_schema=dict(
                    fielddate=qt.String,
                    user_type=qt.String,
                    ui=qt.String,
                    period=qt.String,
                    actions_count=qt.String,
                    action_type=qt.String,
                    returned=qt.UInt64,
                ))(
                    BracketRetentionReducer(action_type, calc_only_days, date=self.date, dateend=self.dateend, key_fields=['ui'])
                ),
                intensity=500,  # data_size_per_job ~= 2-3Mb
            ) \
            .groupby(*KEY_FIELDS) \
            .aggregate(
                users=na.count(),
                returned=na.sum('returned')
            ) \
            .project(
                ne.all(),
                retention=ne.custom(lambda x, y: float(x) / y, 'returned', 'users').with_type(float)
            )
        return self.publish(
            output,
            self.get_statface_report('Collections/Metrics/Retention/BracketRetentionV2'),
            allow_change_job=True
        )

    @require('ActiveUsers.get_created_boards')
    def active_users_boards(self, streams):
        """Считает активную аудиторию, которые за период N дней совершили K действий:
        * создали любую коллекцию
        * создали публичную коллекцию
        * создали corgie_v1 коллекцию
        * создали corgie_v2 коллекцию
        * создали corgie_v3 коллекцию
        """
        yield self.calc_active_users(
            streams['ActiveUsers.get_created_boards'],
            'all_boards'
        )

        yield self.calc_active_users(
            streams['ActiveUsers.get_created_boards'].filter(
                nf.equals('is_private', False)
            ),
            'public_boards'
        )

        yield self.calc_active_users(
            streams['ActiveUsers.get_created_boards'].filter(
                nf.equals('is_corgie_v1', True)
            ),
            'corgie_v1_boards'
        )

        yield self.calc_active_users(
            streams['ActiveUsers.get_created_boards'].filter(
                nf.equals('is_corgie_v2', True)
            ),
            'corgie_v2_boards'
        )

        yield self.calc_active_users(
            streams['ActiveUsers.get_created_boards'].filter(
                nf.equals('is_corgie_v3', True)
            ),
            'corgie_v3_boards'
        )
        # first corgie
        corgie_v1_first_boards = streams['ActiveUsers.get_created_boards'].filter(nf.equals('is_corgie_v1', True)) \
            .groupby('user_id') \
            .top(1, by='fielddate', mode='min')
        corgie_v2_first_boards = streams['ActiveUsers.get_created_boards'].filter(nf.equals('is_corgie_v2', True)) \
            .groupby('user_id') \
            .top(1, by='fielddate', mode='min')
        corgie_v3_first_boards = streams['ActiveUsers.get_created_boards'].filter(nf.equals('is_corgie_v3', True)) \
            .groupby('user_id') \
            .top(1, by='fielddate', mode='min')

        yield self.calc_active_users(
            corgie_v1_first_boards,
            'corgie_v1_first_boards'
        )

        yield self.calc_active_users(
            corgie_v2_first_boards,
            'corgie_v2_first_boards'
        )

        yield self.calc_active_users(
            corgie_v3_first_boards,
            'corgie_v3_first_boards'
        )

        # non first_corgie
        yield self.calc_active_users(
            streams['ActiveUsers.get_created_boards'].filter(
                nf.equals('is_corgie_v1', True)
            ).join(corgie_v1_first_boards, by='board_id', type='left_only'),
            'corgie_v1_nonfirst_boards'
        )

        yield self.calc_active_users(
            streams['ActiveUsers.get_created_boards'].filter(
                nf.equals('is_corgie_v2', True)
            ).join(corgie_v2_first_boards, by='board_id', type='left_only'),
            'corgie_v2_nonfirst_boards'
        )

        yield self.calc_active_users(
            streams['ActiveUsers.get_created_boards'].filter(
                nf.equals('is_corgie_v3', True)
            ).join(corgie_v3_first_boards, by='board_id', type='left_only'),
            'corgie_v3_nonfirst_boards'
        )

    @require('ActiveUsers.get_created_cards')
    def active_users_cards(self, streams):
        """Считает активную аудиторию, которые за период N дней совершили K действий:
        * создали любую карточки
        * создали публичную карточку
        * создали карточку из corgie_v1 коллекции
        * создали карточку из corgie_v2 коллекции
        * создали карточку из corgie_v3 коллекции
        """
        yield self.calc_active_users(
            streams['ActiveUsers.get_created_cards'],
            'all_cards'
        )

        yield self.calc_active_users(
            streams['ActiveUsers.get_created_cards'].filter(
                nf.equals('is_private', False)
            ),
            'public_cards'
        )

        yield self.calc_active_users(
            streams['ActiveUsers.get_created_cards'].filter(
                nf.equals('is_corgie_v1', True)
            ),
            'corgie_v1_cards'
        )

        yield self.calc_active_users(
            streams['ActiveUsers.get_created_cards'].filter(
                nf.equals('is_corgie_v2', True)
            ),
            'corgie_v2_cards'
        )

        yield self.calc_active_users(
            streams['ActiveUsers.get_created_cards'].filter(
                nf.equals('is_corgie_v3', True)
            ),
            'corgie_v3_cards'
        )

    @require('ActiveUsers.get_created_cards', 'ActiveUsers.get_created_boards')
    def active_users_cards_boards(self, streams):
        """Считает активную аудиторию, которые за период N дней совершили K действий:
        * создали любую карточку или коллекцию
        * создали публичную карточку или коллекцию
        * создали карточку или коллекцию corgie_v1
        * создали карточку или коллекцию corgie_v2
        * создали карточку или коллекцию corgie_v3
        """
        yield self.calc_active_users(
            streams['ActiveUsers.get_created_boards'].concat(streams['ActiveUsers.get_created_cards']),
            'all_cards_boards'
        )

        yield self.calc_active_users(
            streams['ActiveUsers.get_created_boards'].concat(streams['ActiveUsers.get_created_cards']).filter(
                nf.equals('is_private', False)
            ),
            'public_cards_boards'
        )

        yield self.calc_active_users(
            streams['ActiveUsers.get_created_boards'].concat(streams['ActiveUsers.get_created_cards']).filter(
                nf.equals('is_corgie_v1', True)
            ),
            'corgie_v1_cards_boards'
        )

        yield self.calc_active_users(
            streams['ActiveUsers.get_created_boards'].concat(streams['ActiveUsers.get_created_cards']).filter(
                nf.equals('is_corgie_v2', True)
            ),
            'corgie_v2_cards_boards'
        )

        yield self.calc_active_users(
            streams['ActiveUsers.get_created_boards'].concat(streams['ActiveUsers.get_created_cards']).filter(
                nf.equals('is_corgie_v3', True)
            ),
            'corgie_v3_cards_boards'
        )

    @require(
        'ActiveUsers.get_visits',
        'EntryPointsUsefulSessions.get_useful_sessions',
    )
    def active_users_visits(self, streams):
        """Считает активную аудиторию, которые за период N дней совершили K действий:
        * совершили визит
        * совершили визит будучи залогином
        * совершили визит с полезным действием
        """
        yandexuid_visits = streams['ActiveUsers.get_visits'] \
            .filter(qf.nonzero('yandexuid')) \
            .project(
                'fielddate',
                user_id='yandexuid',
                ui='ui',
                user_type=ne.const('yandexuid').with_type(str),
            )
        yield self.calc_active_users(yandexuid_visits, 'visits', calc_only_days=True)

        puid_visits = streams['ActiveUsers.get_visits'] \
            .filter(qf.nonzero('puid')) \
            .project(
                'fielddate',
                user_id='puid',
                ui='ui',
                user_type=ne.const('puid').with_type(str),
            )
        yield self.calc_active_users(puid_visits, 'visits', calc_only_days=True)

        useful_sessions = streams['EntryPointsUsefulSessions.get_useful_sessions'] \
            .filter(
                nf.and_(
                    qf.nonzero('yandexuid'),
                    nf.custom(lambda x: x == 1, 'useful_sessions'),
                )
            ) \
            .project(
                'fielddate',
                user_id='yandexuid',
                ui='ui',
                user_type=ne.const('yandexuid').with_type(str),
            )
        yield self.calc_active_users(useful_sessions, 'useful_sessions', calc_only_days=True)

    @require('ActiveUsers.get_created_boards')
    def retention_boards(self, streams):
        """Считает активную аудиторию, которые за период N дней совершили K действий:
        * создали любую коллекцию
        * создали публичную коллекцию
        * создали corgie_v1 коллекцию
        * создали corgie_v2 коллекцию
        * создали corgie_v3 коллекцию
        """
        yield self.calc_bracket_retention(
            streams['ActiveUsers.get_created_boards'],
            'all_boards'
        )

        yield self.calc_bracket_retention(
            streams['ActiveUsers.get_created_boards'].filter(
                nf.equals('is_private', False)
            ),
            'public_boards'
        )

        yield self.calc_bracket_retention(
            streams['ActiveUsers.get_created_boards'].filter(
                nf.equals('is_corgie_v1', True)
            ),
            'corgie_v1_boards'
        )

        yield self.calc_bracket_retention(
            streams['ActiveUsers.get_created_boards'].filter(
                nf.equals('is_corgie_v2', True)
            ),
            'corgie_v2_boards'
        )

        yield self.calc_bracket_retention(
            streams['ActiveUsers.get_created_boards'].filter(
                nf.equals('is_corgie_v3', True)
            ),
            'corgie_v3_boards'
        )

    @require('ActiveUsers.get_created_cards')
    def retention_cards(self, streams):
        """Считает активную аудиторию, которые за период N дней совершили K действий:
        * создали любую карточки
        * создали публичную карточку
        * создали карточку из corgie_v1 коллекции
        * создали карточку из corgie_v2 коллекции
        * создали карточку из corgie_v3 коллекции
        """
        yield self.calc_bracket_retention(
            streams['ActiveUsers.get_created_cards'],
            'all_cards'
        )

        yield self.calc_bracket_retention(
            streams['ActiveUsers.get_created_cards'].filter(
                nf.equals('is_private', False)
            ),
            'public_cards'
        )

        yield self.calc_bracket_retention(
            streams['ActiveUsers.get_created_cards'].filter(
                nf.equals('is_corgie_v1', True)
            ),
            'corgie_v1_cards'
        )

        yield self.calc_bracket_retention(
            streams['ActiveUsers.get_created_cards'].filter(
                nf.equals('is_corgie_v2', True)
            ),
            'corgie_v2_cards'
        )

        yield self.calc_bracket_retention(
            streams['ActiveUsers.get_created_cards'].filter(
                nf.equals('is_corgie_v3', True)
            ),
            'corgie_v3_cards'
        )

    @require('ActiveUsers.get_created_cards', 'ActiveUsers.get_created_boards')
    def retention_cards_boards(self, streams):
        """Считает активную аудиторию, которые за период N дней совершили K действий:
        * создали любую карточку или коллекцию
        * создали публичную карточку или коллекцию
        * создали карточку или коллекцию corgie_v1
        * создали карточку или коллекцию corgie_v2
        * создали карточку или коллекцию corgie_v3
        """
        yield self.calc_bracket_retention(
            streams['ActiveUsers.get_created_boards'].concat(streams['ActiveUsers.get_created_cards']),
            'all_cards_boards'
        )

        yield self.calc_bracket_retention(
            streams['ActiveUsers.get_created_boards'].concat(streams['ActiveUsers.get_created_cards']).filter(
                nf.equals('is_private', False)
            ),
            'public_cards_boards'
        )

        yield self.calc_bracket_retention(
            streams['ActiveUsers.get_created_boards'].concat(streams['ActiveUsers.get_created_cards']).filter(
                nf.equals('is_corgie_v1', True)
            ),
            'corgie_v1_cards_boards'
        )

        yield self.calc_bracket_retention(
            streams['ActiveUsers.get_created_boards'].concat(streams['ActiveUsers.get_created_cards']).filter(
                nf.equals('is_corgie_v2', True)
            ),
            'corgie_v2_cards_boards'
        )

        yield self.calc_bracket_retention(
            streams['ActiveUsers.get_created_boards'].concat(streams['ActiveUsers.get_created_cards']).filter(
                nf.equals('is_corgie_v3', True)
            ),
            'corgie_v3_cards_boards'
        )

    @require(
        'ActiveUsers.get_visits',
        'EntryPointsUsefulSessions.get_useful_sessions'
    )
    def retention_visits(self, streams):
        """Считает активную аудиторию, которые за период N дней совершили K действий:
        * совершили визит
        * совершили визит будучи залогином
        * совершили визит с полезным действием
        """
        yandexuid_visits = streams['ActiveUsers.get_visits'] \
            .filter(qf.nonzero('yandexuid')) \
            .project(
                'fielddate',
                user_id='yandexuid',
                ui='ui',
                user_type=ne.const('yandexuid').with_type(str),
            )
        yield self.calc_bracket_retention(yandexuid_visits, 'visits', calc_only_days=True)

        puid_visits = streams['ActiveUsers.get_visits'] \
            .filter(qf.nonzero('puid')) \
            .project(
                'fielddate',
                user_id='puid',
                ui='ui',
                user_type=ne.const('puid').with_type(str),
            )
        yield self.calc_bracket_retention(puid_visits, 'visits', calc_only_days=True)

        useful_sessions = streams['EntryPointsUsefulSessions.get_useful_sessions'] \
            .filter(
                nf.and_(
                    qf.nonzero('yandexuid'),
                    nf.custom(lambda x: x == 1, 'useful_sessions'),
                )
            ) \
            .project(
                'fielddate',
                user_id='yandexuid',
                ui='ui',
                user_type=ne.const('yandexuid').with_type(str),
            )
        yield self.calc_bracket_retention(useful_sessions, 'useful_sessions', calc_only_days=True)

    @require(
        'ActiveUsers.get_own_cards_accesses',
        'ActiveUsers.get_own_boards_accesses'
    )
    def own_content_retention(self, streams):
        cards = streams['ActiveUsers.get_own_cards_accesses']
        boards = streams['ActiveUsers.get_own_boards_accesses']
        action_name_stream_map = {
            'own_content': boards.concat(cards),
            'own_boards': boards,
            'own_cards': cards
        }
        for action_name, stream in action_name_stream_map.items():
            for user_type in ('yandexuid', 'puid'):
                own_content = stream  \
                    .project(
                        'fielddate',
                        user_id=user_type,
                        ui='ui',
                        user_type=ne.const(user_type).with_type(str),
                    )
                yield self.calc_bracket_retention(
                    own_content,
                    action_name
                )


class ActiveAuthors(Plot):
    """Расчёт активной аудитории по созданию контента в срезе по origin client, ui"""

    def calc_active_authors(self, stream, action_type):
        """Универсальный код вычисления активных авторов
        Заданным окном `назад` считает количество людей, которые совершили N действий за период K дней"""
        KEY_FIELDS = ('fielddate', 'user_type', 'origin_client', 'origin_ui', 'origin_action', 'action_type', 'actions_count', 'period')

        return stream \
            .map(with_hints(output_schema=extended_schema())(
                AddTotalsMapper(['user_type', 'origin_client', 'origin_ui', 'origin_action'], ['fielddate', 'user_id'])
            )) \
            .groupby('user_id', 'user_type', 'origin_client', 'origin_ui', 'origin_action') \
            .reduce(
                with_hints(output_schema=dict(
                    fielddate=qt.String,
                    user_type=qt.String,
                    origin_client=qt.String,
                    origin_ui=qt.String,
                    origin_action=qt.String,
                    period=qt.String,
                    actions_count=qt.String,
                    action_type=qt.String,
                    actions=qt.UInt64,
                ))(
                    ActiveUsersReducer(action_type, date=self.date, dateend=self.dateend, key_fields=['origin_client', 'origin_ui', 'origin_action'])
                ),
                intensity=500,  # data_size_per_job ~= 2-3Mb
            ) \
            .groupby(*KEY_FIELDS) \
            .aggregate(
                users=na.count(),
                actions=na.sum('actions'),
            ) \
            .publish(self.get_statface_report('Collections/Metrics/authors/ActiveAuthorsV4'), allow_change_job=True)

    @require(
        'ActiveUsers.get_created_boards',
        'ActiveUsers.get_created_cards',
    )
    def active_authors(self, streams):
        """Считает активных авторов, которые за период N дней создали K разного контента
        * создали любую коллекцию
        * создали публичную коллекцию
        * создали corgie_v1 коллекцию
        * создали любую карточку или коллекцию
        * создали публичную карточку или коллекцию
        * создали corgie_v1 карточку или коллекцию
        """

        # cards:

        yield self.calc_active_authors(
            streams['ActiveUsers.get_created_cards'],
            'all_cards'
        )

        yield self.calc_active_authors(
            streams['ActiveUsers.get_created_cards'].filter(
                nf.equals('is_private', False)
            ),
            'public_cards'
        )

        yield self.calc_active_authors(
            streams['ActiveUsers.get_created_cards'].filter(
                nf.equals('is_corgie_v1', True)
            ),
            'corgie_v1_cards'
        )

        yield self.calc_active_authors(
            streams['ActiveUsers.get_created_cards'].filter(
                nf.equals('is_corgie_v2', True)
            ),
            'corgie_v2_cards'
        )

        yield self.calc_active_authors(
            streams['ActiveUsers.get_created_cards'].filter(
                nf.equals('is_corgie_v3', True)
            ),
            'corgie_v3_cards'
        )

        # boards:

        yield self.calc_active_authors(
            streams['ActiveUsers.get_created_boards'],
            'all_boards'
        )

        yield self.calc_active_authors(
            streams['ActiveUsers.get_created_boards'].filter(
                nf.equals('is_private', False)
            ),
            'public_boards'
        )

        yield self.calc_active_authors(
            streams['ActiveUsers.get_created_boards'].filter(
                nf.equals('is_corgie_v1', True)
            ),
            'corgie_v1_boards'
        )

        yield self.calc_active_authors(
            streams['ActiveUsers.get_created_boards'].filter(
                nf.equals('is_corgie_v2', True)
            ),
            'corgie_v2_boards'
        )

        yield self.calc_active_authors(
            streams['ActiveUsers.get_created_boards'].filter(
                nf.equals('is_corgie_v3', True)
            ),
            'corgie_v3_boards'
        )

        # cards boards:

        yield self.calc_active_authors(
            streams['ActiveUsers.get_created_boards'].concat(streams['ActiveUsers.get_created_cards']).filter(
                nf.equals('is_private', False)
            ),
            'public_cards_boards'
        )

        yield self.calc_active_authors(
            streams['ActiveUsers.get_created_boards'].concat(streams['ActiveUsers.get_created_cards']).filter(
                nf.equals('is_corgie_v1', True)
            ),
            'corgie_v1_cards_boards'
        )

        yield self.calc_active_authors(
            streams['ActiveUsers.get_created_boards'].concat(streams['ActiveUsers.get_created_cards']).filter(
                nf.equals('is_corgie_v2', True)
            ),
            'corgie_v2_cards_boards'
        )

        yield self.calc_active_authors(
            streams['ActiveUsers.get_created_boards'].concat(streams['ActiveUsers.get_created_cards']).filter(
                nf.equals('is_corgie_v3', True)
            ),
            'corgie_v3_cards_boards'
        )
