# coding: utf-8

from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from analytics.plotter_lib.utils import AddTotalsMapper, PrecomputeTotalsMapper
from analytics.plotter_lib.utils import get_dts_delta, date_range, split_by_dates
from analytics.collections.plotter_collections.plots.utils import readable_shows_location


from nile.api.v1 import (
    with_hints,
    extended_schema,
    filters as nf,
    extractors as ne,
    aggregators as na,
    grouping
)
from qb2.api.v1 import (
    filters as qf,
    typing as qt
)


def get_owner_type(is_organic, is_partner):
    if is_partner is not None and is_partner:
        return 'partner'
    elif is_organic is not None:
        return 'organic' if is_organic else 'toloker'
    else:
        return 'empty'


def get_place_name(*args):
    return '-'.join([str(x) for x in args if x is not None])


def get_content_freshness_threshhold(fielddate, content_created_date):
    if not fielddate or not content_created_date:
        return 'empty'
    dt_delta = get_dts_delta(fielddate, content_created_date)
    if dt_delta.days <= 7:
        return '<=1w'
    if dt_delta.days <= 14:
        return '<=2w'
    if dt_delta.days <= 30:
        return '<=1m'
    if dt_delta.days <= 91:
        return '<=3m'
    if dt_delta.days <= 183:
        return '<=6m'
    if dt_delta.days <= 365:
        return '<=1y'
    return '>1y'


def extract_unique_board_shows_from_sessions_reducer(groups):
    for key, records in groups:
        unique_board_shows = {}
        for r in records:
            if r.board_id and r.board_id in unique_board_shows:
                continue
            unique_board_shows[r.board_id] = r

        for board_id, show in unique_board_shows.items():
            yield show


class AllServicesCollectionsShows(Plot):
    @require('all-collections-shows-log')
    def log(self):
        return {}

    @require(
        'CollectionsShows.all_cards_boards',
        'WebUserSessionsSearch.collections_shows',
        'ImagesUserSessionsSearch.collections_shows',
        'MordaUserSessionsSearch.collections_shows',
        layer='sessions',
    )
    def all(self, streams):
        """Выжимки показов карточек/бордов со всех поверхностей.
        Не включает в себя показы страниц профиля"""
        return self.job.concat(*streams.values()) \
            .sort('fielddate', 'service', 'ui', 'page_name', 'location', 'yandexuid', 'reqid', 'timestamp') \
            .checkpoint(self.get_checkpoint_name('all_shows_from_logs'))

    @require(
        'AllServicesCollectionsShows.all',
        'PrepCards.with_organic_flag',
        layer='sessions',
    )
    def cards_shows_joined_with_db(self, streams):
        cards = streams['AllServicesCollectionsShows.all'] \
            .filter(nf.equals('content_type', 'card')) \
            .project(ne.all('board_id'))  # board_id будем для всех карточек брать из приджоиненой базы

        cards_data = streams['PrepCards.with_organic_flag'] \
            .project(
                'card_id',
                content_owner_id='card_owner_id',
                is_owner_organic='card_owner_is_organic',
                is_owner_partner='card_owner_is_partner',
                is_private='card_is_private',
                is_banned='card_is_banned',
                board_id='card_board_id',
                content_created_date='card_created_at_date',
                content_created_timestamp='card_created_at_timestamp',
                is_card_product=ne.custom(lambda x: x == 'product' if x else False, 'card_meta_info_type').with_type(bool),
                content_owner_puid=ne.custom(lambda x: None if x is None else ('p' + x), 'content_owner_puid').with_type(qt.Optional[qt.String]),
                content_owner_company_id=ne.custom(lambda x: None if x is None else x, 'user_company_id').with_type(qt.Optional[qt.String]),
            )

        return cards \
            .join(cards_data, type='left', by='card_id', allow_undefined_keys=False)

    @require(
        'AllServicesCollectionsShows.all',
        'PrepBoards.with_organic_flag',
        layer='sessions',
    )
    def boards_shows_joined_with_db(self, streams):
        boards = streams['AllServicesCollectionsShows.all'].filter(
            nf.equals('content_type', 'board')
        )

        boards_data = streams['PrepBoards.with_organic_flag'] \
            .project(
                'board_id',
                'board_url',
                content_owner_id='board_owner_id',
                is_owner_organic='board_owner_is_organic',
                is_owner_partner='board_owner_is_partner',
                is_private='board_is_private',
                is_banned='board_is_banned',
                content_created_date='board_created_at_date',
                content_created_timestamp='board_created_at_timestamp',
                content_owner_puid=ne.custom(lambda x: None if x is None else ('p' + x), 'content_owner_puid').with_type(qt.Optional[qt.String]),
                content_owner_company_id=ne.custom(lambda x: None if x is None else x, 'user_company_id').with_type(qt.Optional[qt.String]),
            )

        return boards \
            .join(boards_data, type='left', by='board_id', allow_undefined_keys=False)

    @require(
        'AllServicesCollectionsShows.cards_shows_joined_with_db',
        'AllServicesCollectionsShows.boards_shows_joined_with_db',
        'CollectionsShows.users_shows_joined_with_db',
        'Corgie.corgie',
        layer='sessions',
    )
    def joined_with_db(self, streams):
        """Все выжимки показов карточек/бордов/профилей с инфой из базы"""
        corgie = streams['Corgie.corgie'] \
            .project(
                'board_id',
                is_corgie_board=ne.const(True),
            )

        stream = self.job.concat(
            streams['AllServicesCollectionsShows.cards_shows_joined_with_db'],
            streams['AllServicesCollectionsShows.boards_shows_joined_with_db'],
            streams['CollectionsShows.users_shows_joined_with_db'],
            ) \
            .join(corgie, type='left', by='board_id', allow_undefined_keys=True) \
            .checkpoint(self.get_checkpoint_name('all_shows_joined_with_db'))

        for table_stream in split_by_dates(
            stream,
            [x.strftime(DATE_FORMAT) for x in date_range(self.date, self.dateend)],
            '//home/collections/analytics/backups/COLA-10_collections_content_shows_squeeze/',
            sort_by=['fielddate', 'service', 'ui', 'page_name', 'location', 'content_subtype', 'yandexuid', 'reqid', 'timestamp'],
        ):
            yield table_stream
        yield stream


class AllServicesCollectionsShowsReports(Plot):
    @require(
        'AllServicesCollectionsShows.joined_with_db',
        layer='sessions',
    )
    def prepare_data(self, streams):
        return streams['AllServicesCollectionsShows.joined_with_db'] \
            .project(
                'fielddate',
                place=ne.custom(get_place_name, 'service', 'ui', 'page_name', 'location', 'content_subtype').with_type(str),
                content_type=ne.custom(lambda x: 'empty' if x is None else x, 'content_type').with_type(str),
                is_corgie_board=ne.custom(lambda x: 'False' if x is None else str(x), 'is_corgie_board').with_type(str),
                freshness=ne.custom(get_content_freshness_threshhold, 'fielddate', 'content_created_date').with_type(str),
                owner_type=ne.custom(get_owner_type, 'is_owner_organic', 'is_owner_partner').with_type(str),
            )

    def prepare_and_publish(self, stream, key_field, report_path):
        REPORT_KEY_FIELDS = ('fielddate', 'place', 'content_type', key_field)

        return stream \
            .sort(*REPORT_KEY_FIELDS) \
            .map(
                with_hints(output_schema=extended_schema(hits=int))(
                    PrecomputeTotalsMapper(REPORT_KEY_FIELDS)
                ),
                ordered=True
            ) \
            .map(with_hints(output_schema=extended_schema())(
                AddTotalsMapper(REPORT_KEY_FIELDS, ['fielddate', 'hits'])
            )) \
            .groupby(*REPORT_KEY_FIELDS) \
            .aggregate(
                hits=na.sum('hits')
            ) \
            .publish(self.get_statface_report(report_path), allow_change_job=True)

    @require('AllServicesCollectionsShowsReports.prepare_data', layer='sessions')
    def report_users(self, streams):
        return self.prepare_and_publish(
            streams['AllServicesCollectionsShowsReports.prepare_data'],
            'owner_type',
            'Collections/Metrics/sessions/AllCollectionsShows_UserTypes',
        )

    @require('AllServicesCollectionsShowsReports.prepare_data', layer='sessions')
    def report_corgie(self, streams):
        return self.prepare_and_publish(
            streams['AllServicesCollectionsShowsReports.prepare_data'],
            'is_corgie_board',
            'Collections/Metrics/sessions/AllCollectionsShows_Corgie',
        )

    @require('AllServicesCollectionsShowsReports.prepare_data', layer='sessions')
    def report_freshness(self, streams):
        return self.prepare_and_publish(
            streams['AllServicesCollectionsShowsReports.prepare_data'],
            'freshness',
            'Collections/Metrics/sessions/AllCollectionsShows_Freshness',
        )

    @require(
        'AllServicesCollectionsShows.joined_with_db',
        layer='sessions',
    )
    def report_reqids_count(self, streams):
        """Отчёт по количеству reqid по всем поверхностям
        За единицу показа считается целиком страница, или 1 показ колдунщика"""
        REPORT_KEY_FIELDS = ['fielddate', 'service', 'ui', 'page_name']

        return streams['AllServicesCollectionsShows.joined_with_db'] \
            .filter(
                nf.custom(lambda page_name: False if page_name is None or page_name == '' else True, 'page_name'),
            ) \
            .unique(*(REPORT_KEY_FIELDS + ['reqid']), keep_only_group_fields=True) \
            .sort(*REPORT_KEY_FIELDS) \
            .map(
                with_hints(output_schema=extended_schema(hits=int))(
                    PrecomputeTotalsMapper(REPORT_KEY_FIELDS)
                ),
                ordered=True
            ) \
            .map(with_hints(output_schema=extended_schema())(
                AddTotalsMapper(REPORT_KEY_FIELDS, ['fielddate', 'hits'])
            )) \
            .groupby(*REPORT_KEY_FIELDS) \
            .aggregate(
                hits=na.sum('hits')
            ) \
            .publish(self.get_statface_report('Collections/Metrics/sessions/AllCollectionsShows_Reqids'), allow_change_job=True)

    @require('AllServicesCollectionsShows.joined_with_db', 'PrepBoards.with_organic_flag', layer='sessions')
    def boards_with_shows_clicks(self, streams):
        REPORT_KEY_FIELDS = (
            'fielddate',
            'location',
            'board_features_corgie_v2',
            'board_features_corgie_v3',
            'board_moder_status',
            'user_type',
            'ui'
        )
        boards = streams['PrepBoards.with_organic_flag'] \
            .project(
                'board_id',
                'user_type',
                board_features_corgie_v2=ne.custom(lambda x: x if x else 'empty', 'board_features_corgie_v2').with_type(str),
                board_features_corgie_v3=ne.custom(lambda x: x if x else 'empty', 'board_features_corgie_v3').with_type(str),
                board_moder_status=ne.custom(lambda x: str(x) if x else 'empty', 'board_moder_status').with_type(str),

            )

        return streams['AllServicesCollectionsShows.joined_with_db'] \
            .groupby('yandexuid') \
            .sort('timestamp') \
            .groupby(grouping.sessions()) \
            .reduce(with_hints(output_schema=extended_schema())(extract_unique_board_shows_from_sessions_reducer)) \
            .filter(qf.nonzero('board_id')) \
            .project(
                'board_id',
                'fielddate',
                ui=ne.custom(lambda x: x if x else 'empty', 'ui').with_type(str),
                location=ne.custom(readable_shows_location, 'service', 'page_name', 'location', 'content_type', 'content_subtype').with_type(str)
            ) \
            .join(boards, type='inner', by='board_id') \
            .sort(*REPORT_KEY_FIELDS) \
            .map(
                with_hints(output_schema=extended_schema(hits=int))(
                    PrecomputeTotalsMapper(REPORT_KEY_FIELDS)
                ),
                ordered=True
            ) \
            .map(with_hints(output_schema=extended_schema())(
                AddTotalsMapper(REPORT_KEY_FIELDS, ['fielddate', 'hits'])
            )) \
            .groupby(*REPORT_KEY_FIELDS) \
            .aggregate(
                shows=na.sum('hits')
            ) \
            .publish(self.get_statface_report('Collections/Metrics/sessions/AllCollectionsShows_CorgieV2'))
