# coding: utf-8

from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from analytics.plotter_lib.utils import date_range
from nile.api.v1 import (
    extractors as ne,
    aggregators as na,
    filters as nf,
    Record,
    with_hints
)
from qb2.api.v1 import (
    filters as qf,
)


class AccumReducer(object):
    def __init__(self, date, dateend):
        self.date = date
        self.dateend = dateend

    def __call__(self, groups):
        last_fielddate = ''
        authors = 0
        date_authors = {}
        for key, records in groups:
            for r in records:
                if r.get('authors'):
                    authors += r.authors
                if last_fielddate != r.get('fielddate'):
                    last_fielddate = r.fielddate
                    date_authors[r.fielddate] = authors

        authors = None
        for date in date_range(self.date, self.dateend):
            if date.strftime(DATE_FORMAT) in date_authors:
                authors = date_authors[date.strftime(DATE_FORMAT)]
            if authors is not None:
                yield Record(authors=authors, fielddate=date.strftime(DATE_FORMAT))


@with_hints(output_schema=dict(authors=int, fielddate=str))
def accum_reducer(groups):
    last_fielddate = ''
    authors = 0
    date_authors = {}
    for key, records in groups:
        for r in records:
            if r.get('authors'):
                authors += r.authors
            if last_fielddate != r.get('fielddate'):
                last_fielddate = r.fielddate
                date_authors[r.fielddate] = authors


class Authors(Plot):
    def prep_author_of_cards_data(self, stream, require, authors_type):
        cards = stream[require].project(
            'card_owner_id',
            fielddate='card_created_at_date'
        )
        date_start = self.date.strftime(DATE_FORMAT)
        date_end = self.dateend.strftime(DATE_FORMAT)

        return cards.groupby('fielddate') \
            .aggregate(authors=na.count_distinct('card_owner_id')) \
            .project('fielddate', 'authors', authors_type=ne.const(authors_type)) \
            .filter(nf.custom(lambda x: date_start <= x <= date_end, 'fielddate'))

    @require('PrepCards.without_bookmarks')
    def authors_of_cards(self, streams):
        return self.prep_author_of_cards_data(streams, 'PrepCards.without_bookmarks', 'authors_of_cards') \
            .publish(self.get_statface_report('Collections/Metrics/authors/AuthorsCommonReportV1'), allow_change_job=True)

    @require('PrepCards.organic_cards_without_spam')
    def author_of_organic_cards_without_spam(self, streams):
        return self.prep_author_of_cards_data(streams, 'PrepCards.organic_cards_without_spam', 'authors_of_organic_cards_without_spam') \
            .publish(self.get_statface_report('Collections/Metrics/authors/AuthorsCommonReportV1'), allow_change_job=True)

    @require('Corgie.corgie')
    def authors_of_corgie(self, streams):
        corgie = streams['Corgie.corgie'].project(
            'board_owner_id',
            fielddate='good_board_fielddate'
        )

        return corgie.groupby('fielddate') \
            .aggregate(authors=na.count_distinct('board_owner_id')) \
            .project('fielddate', 'authors', authors_type=ne.const('authors_of_corgie')) \
            .publish(self.get_statface_report('Collections/Metrics/authors/AuthorsCommonReportV1'), allow_change_job=True)

    @require('Corgie.corgie')
    def authors_of_corgie_v3(self, streams):
        date_start = self.date.strftime(DATE_FORMAT)
        date_end = self.dateend.strftime(DATE_FORMAT)
        corgie3 = streams['Corgie.corgie'] \
            .filter(
                nf.or_(
                    nf.equals('board_features_corgie_v3', '1'),
                    nf.equals('board_features_corgie_v3', '2'),
                    nf.equals('board_features_corgie_v3', '3')
                )
            ) \
            .project('board_owner_id', 'board_features_corgie_v3', fielddate='good_board_fielddate') \
            .filter(nf.custom(lambda dt: date_start <= dt <= date_end, 'fielddate'))

        for corgie_type in ['all', '1', '2', '3']:
            if corgie_type == 'all':
                yield corgie3.groupby('fielddate') \
                    .aggregate(
                        authors=na.count_distinct('board_owner_id')
                    )\
                    .project(ne.all(), authors_type=ne.const('authors_of_corgie_v3'))\
                    .publish(self.get_statface_report('Collections/Metrics/authors/AuthorsCommonReportV1'), allow_change_job=True)
            else:
                yield corgie3 \
                    .filter(nf.equals('board_features_corgie_v3', corgie_type)) \
                    .groupby('fielddate') \
                    .aggregate(
                        authors=na.count_distinct('board_owner_id')
                    ) \
                    .project(ne.all(), authors_type=ne.const('authors_of_corgie_v3_{}'.format(corgie_type))) \
                    .publish(self.get_statface_report('Collections/Metrics/authors/AuthorsCommonReportV1'),
                             allow_change_job=True)

    @require('CollectionsRedirLog.parsed')
    def yandex_users(self, streams):
        return streams['CollectionsRedirLog.parsed'] \
            .filter(nf.equals('ar', '16')) \
            .unique('fielddate', 'puid', keep_only_group_fields=True) \
            .groupby('fielddate') \
            .aggregate(authors=na.count()) \
            .project('fielddate', 'authors', authors_type=ne.const('yandex_users')) \
            .publish(self.get_statface_report('Collections/Metrics/authors/AuthorsCommonReportV1'), allow_change_job=True)

    @require('PrepUsers.with_organic_flag', 'PrepBoards.with_organic_flag')
    def companies(self, streams):

        yield streams['PrepUsers.with_organic_flag'] \
            .filter(nf.equals('user_type', 'company')) \
            .project('user_type', fielddate='user_created_date') \
            .groupby('fielddate') \
            .aggregate(authors=na.count()) \
            .groupby() \
            .sort('fielddate') \
            .reduce(with_hints(output_schema=dict(authors=int, fielddate=str))(AccumReducer(self.date, self.dateend))) \
            .project('fielddate', 'authors', authors_type=ne.const('companies_accum')) \
            .publish(self.get_statface_report('Collections/Metrics/authors/AuthorsCommonReportV1'), allow_change_job=True)

        yield streams['PrepBoards.with_organic_flag'] \
            .filter(nf.and_(qf.nonzero('board_features_corgie_v3'), nf.equals('user_type', 'company'))) \
            .groupby('board_owner_id', 'user_type') \
            .aggregate(fielddate=na.min('board_created_at_date')) \
            .groupby('fielddate') \
            .aggregate(
                authors=na.count()) \
            .groupby() \
            .sort('fielddate') \
            .reduce(with_hints(output_schema=dict(authors=int, fielddate=str))(AccumReducer(self.date, self.dateend))) \
            .project('fielddate', 'authors', authors_type=ne.const('companies_with_corgie_v3_accum')) \
            .publish(self.get_statface_report('Collections/Metrics/authors/AuthorsCommonReportV1'), allow_change_job=True)

        yield streams['PrepBoards.with_organic_flag'] \
            .filter(
                nf.and_(
                    nf.not_(nf.equals('board_is_default', True)),
                    nf.custom(lambda x: x >= 3, 'board_public_card_count'),
                    nf.equals('user_type', 'company')
                )
            ) \
            .groupby('board_owner_id', 'user_type') \
            .aggregate(fielddate=na.min('board_created_at_date')) \
            .groupby('fielddate') \
            .aggregate(authors=na.count()) \
            .groupby() \
            .sort('fielddate') \
            .reduce(with_hints(output_schema=dict(authors=int, fielddate=str))(AccumReducer(self.date, self.dateend))) \
            .project('fielddate', 'authors', authors_type=ne.const('companies_with_good_public_board_accum')) \
            .publish(self.get_statface_report('Collections/Metrics/authors/AuthorsCommonReportV1'), allow_change_job=True)
