# coding: utf-8

from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from nile.api.v1 import (
    Record,
    with_hints,
    aggregators as na,
    filters as nf
)


BAN_MASK = {
    0: 'EBR_NONE',
    2 ** 0: 'EBR_COMMON',
    2 ** 1: 'EBR_PORNO',
    2 ** 2: 'EBR_PERVERSION',
    2 ** 3: 'EBR_GRUESOME',
    2 ** 4: 'EBR_COPYRIGHT',
    2 ** 5: 'EBR_OBLIVION',
    2 ** 6: 'EBR_RKN',
    2 ** 16: 'UNKNOWN_REASON',
    2 ** 17: 'CW_PORNO',
    2 ** 18: 'CW_SHOCK',
    2 ** 19: 'CW_VIOLENCE',
    2 ** 20: 'CW_OTHER',
    2 ** 21: 'ANTISPAM_COMMON',
    2 ** 22: 'ANTISPAM_SPAM',
    2 ** 23: 'ANTISPAM_OBSCENE',
    2 ** 24: 'ANTISPAM_NONSENSE',
    2 ** 25: 'ANTISPAM_INSULT',
    2 ** 26: 'ANTISPAM_LAW',
    2 ** 27: 'RKN_URL',
    2 ** 28: 'MANUAL_SOFT',
    2 ** 29: 'MANUAL_HARD',
    2 ** 30: 'OBVIOUS_PORNO',
    2 ** 31: 'OBVIOUS_INAPPROPRIATE',
    2 ** 32: 'FORBIDDEN_URL',
    2 ** 33: 'ANTIFRAUD_SPAM',
    2 ** 34: 'VIDEOPARSE_PORNO',
    2 ** 35: 'PLACEHOLDER_CONTENT',
    2 ** 36: 'VIDEO_PORNO'
}


@with_hints(output_schema=dict(fielddate=str, card_source_type=str, reason=str))
def ban_reason_mapper(recs):
    for rec in recs:
        if not rec.card_is_banned:
            continue

        yield Record(
            fielddate=rec.card_created_at_date,
            card_source_type=rec.card_source_type,
            reason=BAN_MASK.get(rec.card_ban_mask, 'EMPTY')
        )


class CardBans(Plot):
    @require('Cards.parsed')
    def calculate(self, streams):
        cards = streams['Cards.parsed']

        date_start = self.date.strftime(DATE_FORMAT)
        date_end = self.dateend.strftime(DATE_FORMAT)

        card_bans = cards.map(ban_reason_mapper) \
            .groupby('fielddate', 'card_source_type', 'reason') \
            .aggregate(
                bans=na.count()
            ) \
            .filter(nf.custom(lambda x: x <= date_end, 'fielddate'))

        card_bans_stat = card_bans \
            .filter(nf.custom(lambda x: date_start <= x, 'fielddate')) \
            .publish(self.get_statface_report('Collections/Metrics/bans/card_bans'), allow_change_job=True)
        yield 'card_bans_stat', card_bans_stat

        card_bans_accum = card_bans \
            .groupby('card_source_type', 'reason') \
            .aggregate(
                fielddate=na.max('fielddate'),
                bans=na.sum('bans')
            ) \
            .publish(self.get_statface_report('Collections/Metrics/bans/card_bans_accum'), allow_change_job=True)
        yield 'card_bans_accum', card_bans_accum
