# coding: utf-8

from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from nile.api.v1 import (
    Record,
    with_hints,
    aggregators as na,
    filters as nf,
)


@with_hints(output_schema={
    'fielddate': str,
    'book': int,
    'film': int,
    'game': int,
    'geo': int,
    'image': int,
    'link': int,
    'news': int,
    'organization': int,
    'series': int,
    'video': int,
    'image_product': int,
})
def cards_by_source_type_mapper(recs):
    for rec in recs:
        outrec = {
            'fielddate': rec.card_created_at_date,
            'book': 0,
            'film': 0,
            'game': 0,
            'geo': 0,
            'image': 0,
            'link': 0,
            'news': 0,
            'organization': 0,
            'series': 0,
            'video': 0,
            'image_product': 0,
        }

        if rec.card_meta_info_type == 'product':
            outrec['image_product'] = 1
        elif rec.card_source_type:
            outrec[rec.card_source_type] = 1
        else:
            outrec['unknown'] = 1

        yield Record.from_dict(outrec)


@with_hints(output_schema={'fielddate': str, 'product': int})
def accum_product_reducer(groups):
    outrec = {'fielddate': '', 'product': 0}
    for key, recs in groups:
        for rec in recs:
            outrec['fielddate'] = rec.fielddate
            outrec['product'] = rec.image_product + outrec.get('product', 0)

            yield Record.from_dict(outrec)


@with_hints(output_schema={'fielddate': str, 'authors': int})
def accum_author_reducer(groups):
    outrec = {'fielddate': '', 'authors': 0}
    for key, recs in groups:
        for rec in recs:
            outrec['fielddate'] = rec.fielddate
            outrec['authors'] = rec.authors + outrec.get('authors', 0)

            yield Record.from_dict(outrec)


class CardSourceTypeMetrics(Plot):
    @require('PrepCards.without_bookmarks')
    def cards_by_source_type(self, streams):
        date_start = self.date.strftime(DATE_FORMAT)
        date_end = self.dateend.strftime(DATE_FORMAT)

        cards = streams['PrepCards.without_bookmarks'].filter(nf.custom(lambda cdt: cdt <= date_end, 'card_created_at_date'))

        open_card_by_source_type = cards \
            .filter(nf.equals('card_is_private', False)) \
            .map(cards_by_source_type_mapper) \
            .groupby('fielddate') \
            .aggregate(
                book=na.sum('book'),
                film=na.sum('film'),
                game=na.sum('game'),
                geo=na.sum('geo'),
                image=na.sum('image'),
                link=na.sum('link'),
                news=na.sum('news'),
                organization=na.sum('organization'),
                series=na.sum('series'),
                video=na.sum('video'),
                image_product=na.sum('image_product')
            ) \
            .filter(nf.custom(lambda fd: fd >= date_start, 'fielddate')) \
            .publish(self.get_statface_report('Collections/Metrics/source_type_open_card_count'), allow_change_job=True)
        yield 'open_card_by_source_type', open_card_by_source_type

        card_by_source_type = cards \
            .map(cards_by_source_type_mapper) \
            .groupby('fielddate') \
            .aggregate(
                book=na.sum('book'),
                film=na.sum('film'),
                game=na.sum('game'),
                geo=na.sum('geo'),
                image=na.sum('image'),
                link=na.sum('link'),
                news=na.sum('news'),
                organization=na.sum('organization'),
                series=na.sum('series'),
                video=na.sum('video'),
                image_product=na.sum('image_product')
            ) \
            .filter(nf.custom(lambda fd: fd >= date_start, 'fielddate')) \
            .publish(self.get_statface_report('Collections/Metrics/source_type_card_count'), allow_change_job=True)
        yield 'card_by_source_type', card_by_source_type

    @require('PrepCards.organic_cards_without_spam')
    def good_cards_by_source_type(self, streams):
        date_start = self.date.strftime(DATE_FORMAT)
        date_end = self.dateend.strftime(DATE_FORMAT)

        good_cards = streams['PrepCards.organic_cards_without_spam'] \
            .filter(nf.custom(lambda cdt: cdt <= date_end, 'card_created_at_date'))

        products = good_cards \
            .filter(nf.equals('card_meta_info_type', 'product')) \
            .groupby('card_owner_id') \
            .aggregate(
                fielddate=na.min('card_created_at_date')
            ) \
            .groupby('fielddate') \
            .aggregate(
                authors=na.count()
            ) \
            .groupby()     \
            .sort('fielddate') \
            .reduce(accum_author_reducer) \
            .publish(self.get_statface_report('Collections/Metrics/authors_with_product'), allow_change_job=True)
        yield 'products', products

        open_cards_without_spam_by_source_type = good_cards \
            .filter(nf.equals('card_is_private', False)) \
            .map(cards_by_source_type_mapper) \
            .groupby('fielddate') \
            .aggregate(
                book=na.sum('book'),
                film=na.sum('film'),
                game=na.sum('game'),
                geo=na.sum('geo'),
                image=na.sum('image'),
                link=na.sum('link'),
                news=na.sum('news'),
                organization=na.sum('organization'),
                series=na.sum('series'),
                video=na.sum('video'),
                image_product=na.sum('image_product')
            ) \
            .filter(nf.custom(lambda fd: fd >= date_start, 'fielddate')) \
            .publish(self.get_statface_report('Collections/Metrics/source_type_open_cards_without_spam'), allow_change_job=True)
        yield 'open_cards_without_spam_by_source_type', open_cards_without_spam_by_source_type

        good_cards_by_source_type = good_cards \
            .map(cards_by_source_type_mapper) \
            .groupby('fielddate') \
            .aggregate(
                book=na.sum('book'),
                film=na.sum('film'),
                game=na.sum('game'),
                geo=na.sum('geo'),
                image=na.sum('image'),
                link=na.sum('link'),
                news=na.sum('news'),
                organization=na.sum('organization'),
                series=na.sum('series'),
                video=na.sum('video'),
                image_product=na.sum('image_product')
            ) \
            .filter(nf.custom(lambda fd: fd >= date_start, 'fielddate')) \
            .publish(self.get_statface_report('Collections/Metrics/source_type_card_count_without_spam'), allow_change_job=True)
        yield 'good_cards_by_source_type', good_cards_by_source_type

        product_accum = good_cards_by_source_type \
            .groupby() \
            .sort('fielddate') \
            .reduce(accum_product_reducer) \
            .filter(nf.custom(lambda fd: fd >= date_start, 'fielddate')) \
            .publish(self.get_statface_report('Collections/Metrics/product_accum'), allow_change_job=True)
        yield 'product_accum', product_accum
