# coding: utf-8

import datetime

from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from nile.api.v1 import (
    Record,
    with_hints,
    extractors as ne,
    aggregators as na,
    filters as nf
)

from analytics.plotter_lib.utils import AddTotalsMapper


@with_hints(output_schema=dict(fielddate=str, collab_board_count=int))
def accum_reducer(groups):
    for key, recs in groups:
        outrec = {'collab_board_count': 0}
        for rec in recs:
            outrec['fielddate'] = rec.fielddate
            outrec['collab_board_count'] += rec.collab_board_count

            yield Record.from_dict(outrec)


class Collabs(Plot):
    @require(
        'CollectionsRedirLog.parsed',
        'InvitationsBackup.get',
        'Users.parsed',
        'Cards.parsed',
        'Boards.parsed',
        'PrepBoards.collabs',
        'Corgie.corgie',
        'Channels.get'
    )
    def calculate_metrics(self, streams):
        collections_redir_log = streams['CollectionsRedirLog.parsed']
        backup_invitations = streams['InvitationsBackup.get'].unique('board_id')
        users = streams['Users.parsed']
        cards = streams['Cards.parsed']
        boards = streams['Boards.parsed']
        collabs = streams['PrepBoards.collabs']
        corgie = streams['Corgie.corgie']
        channels = streams['Channels.get']

        new_invitations = collections_redir_log \
            .filter(nf.equals('path', 'finish.invitation.accept')) \
            .groupby('board_id') \
            .top(1, by='fielddate', mode='min') \
            .project('fielddate', 'board_id', 'user_id') \
            .join(backup_invitations, by='board_id', type='left_only')

        self.put(new_invitations, '//home/collections/analytics/invitations', append=True)
        yield 'new_invitations', new_invitations

        invitations = new_invitations.concat(backup_invitations)

        collab_board_accum = invitations \
            .groupby('fielddate') \
            .aggregate(
                collab_board_count=na.count()
            ) \
            .project('fielddate', collab_board_count=ne.custom(lambda x: x if x else 0, 'collab_board_count').with_type(int)) \
            .publish(self.get_statface_report('Collections/Metrics/collabs/collab_board_daily'), allow_change_job=True) \
            .groupby() \
            .sort('fielddate') \
            .reduce(accum_reducer) \
            .publish(self.get_statface_report('Collections/Metrics/collabs/collab_board_accum'), allow_change_job=True)
        yield 'collab_board_accum', collab_board_accum

        collab_card_count = invitations \
            .join(cards, by_left='board_id', by_right='card_board_id', type='inner', allow_undefined_keys=False) \
            .filter(nf.custom(lambda invitation_accept, created_at: True if created_at >= invitation_accept else False, 'fielddate', 'card_created_at_datetime')) \
            .groupby('card_created_at_date') \
            .aggregate(
                collab_card_count=na.count(),
                collab_update_count=na.count_distinct('board_id')
            ) \
            .project(
                collab_card_count=ne.custom(lambda x: x if x else 0, 'collab_card_count').with_type(int),
                collab_update_count=ne.custom(lambda x: x if x else 0, 'collab_update_count').with_type(int),
                fielddate='card_created_at_date'
            ) \
            .publish(self.get_statface_report('Collections/Metrics/collabs/collab_card_count'), allow_change_job=True)
        yield 'collab_card_count', collab_card_count

        collab_accum_slices_count = collabs.join(channels.project('board_id', is_channel=ne.const(True)), by='board_id', type='left') \
            .aggregate(
                collab_board_count=na.count(),
                collab_private_board_count=na.count(nf.equals('board_is_private', True)),
                collab_public_board_count=na.count(nf.equals('board_is_private', False)),
                collab_two_more_author_count=na.count(nf.custom(lambda x: True if isinstance(x, list) and len(x) == 2 else False, 'board_permissions')),
                collab_three_more_author_count=na.count(nf.custom(lambda x: True if isinstance(x, list) and 3 <= len(x) < 5 else False, 'board_permissions')),
                collab_five_more_author_count=na.count(nf.custom(lambda x: True if isinstance(x, list) and 5 <= len(x) < 10 else False, 'board_permissions')),
                collab_ten_more_author_count=na.count(nf.custom(lambda x: True if isinstance(x, list) and 10 <= len(x) else False, 'board_permissions')),

                channel_count=na.count(nf.equals('is_channel', True)),
                channel_private_count=na.count(nf.and_(nf.equals('board_is_private', True), nf.equals('is_channel', True))),
                channel_public_count=na.count(nf.and_(nf.equals('board_is_private', False), nf.equals('is_channel', True))),
                channel_two_more_author_count=na.count(nf.and_(nf.custom(lambda x: True if isinstance(x, list) and len(x) == 2 else False, 'board_permissions'), nf.equals('is_channel', True))),
                channel_three_more_author_count=na.count(nf.and_(nf.custom(lambda x: True if isinstance(x, list) and 3 <= len(x) < 5 else False, 'board_permissions'), nf.equals('is_channel', True))),
                channel_five_more_author_count=na.count(nf.and_(nf.custom(lambda x: True if isinstance(x, list) and 5 <= len(x) < 10 else False, 'board_permissions'), nf.equals('is_channel', True))),
                channel_ten_more_author_count=na.count(nf.and_(nf.custom(lambda x: True if isinstance(x, list) and 10 <= len(x) else False, 'board_permissions'), nf.equals('is_channel', True)))
            ) \
            .project(
                collab_board_count=ne.custom(lambda x: x if x else 0, 'collab_board_count').with_type(int),
                collab_private_board_count=ne.custom(lambda x: x if x else 0, 'collab_private_board_count').with_type(int),
                collab_public_board_count=ne.custom(lambda x: x if x else 0, 'collab_public_board_count').with_type(int),
                collab_two_more_author_count=ne.custom(lambda x: x if x else 0, 'collab_two_more_author_count').with_type(int),
                collab_three_more_author_count=ne.custom(lambda x: x if x else 0, 'collab_three_more_author_count').with_type(int),
                collab_five_more_author_count=ne.custom(lambda x: x if x else 0, 'collab_five_more_author_count').with_type(int),
                collab_ten_more_author_count=ne.custom(lambda x: x if x else 0, 'collab_ten_more_author_count').with_type(int),

                channel_count=ne.custom(lambda x: x if x else 0, 'channel_count').with_type(int),
                channel_private_count=ne.custom(lambda x: x if x else 0, 'channel_private_count').with_type(int),
                channel_public_count=ne.custom(lambda x: x if x else 0, 'channel_public_count').with_type(int),
                channel_two_more_author_count=ne.custom(lambda x: x if x else 0, 'channel_two_more_author_count').with_type(int),
                channel_three_more_author_count=ne.custom(lambda x: x if x else 0, 'channel_three_more_author_count').with_type(int),
                channel_five_more_author_count=ne.custom(lambda x: x if x else 0, 'channel_five_more_author_count').with_type(int),
                channel_ten_more_author_count=ne.custom(lambda x: x if x else 0, 'channel_ten_more_author_count').with_type(int),
                fielddate=ne.const(datetime.datetime.now().strftime('%Y-%m-%d')).with_type(str)
            ) \
            .publish(self.get_statface_report('Collections/Metrics/collabs/collab_accum_slices_count'), allow_change_job=True)
        yield 'collab_accum_slices_count', collab_accum_slices_count

        unload_date = (self.dateend - datetime.timedelta(days=2)).strftime(DATE_FORMAT)
        collabs_daily_unload = collabs.join(invitations.filter(nf.equals('fielddate', unload_date)).project('board_id'), type='inner', by='board_id') \
            .join(users, type='inner', by_left='board_owner_id', by_right='user_id') \
            .project(
                'user_login',
                'board_is_private',
                'board_public_card_count',
                'board_private_card_count',
                'board_coauthors_count',
                'board_slug',
                'board_id',
                'user_id',
                'board_description',
                'board_title',
                fielddate=ne.const(unload_date).with_type(str),
                board_url=ne.custom(lambda login, slug: 'https://yandex.ru/collections/user/{}/{}/'.format(login, slug), 'user_login', 'board_slug').with_type(str),
            ) \
            .put('//home/collections/analytics/backups/collabs/{}'.format(unload_date))
        yield 'collabs_daily_unload', collabs_daily_unload

        collab_corgies = collabs.project('board_id') \
            .join(corgie.project('board_id', fielddate='good_board_fielddate'), type='inner', by='board_id') \
            .groupby('fielddate') \
            .aggregate(
                collab_corgies=na.count()
            ) \
            .publish(self.get_statface_report('Collections/Metrics/collabs/collab_corgies'), allow_change_job=True)
        yield 'collab_corgies', collab_corgies

        collabs_uncollabs = collections_redir_log \
            .filter(nf.and_(nf.equals('path', 'finish.coauthor.remove'), nf.equals('type', 'leave'))) \
            .groupby('board_id') \
            .top(1, by='fielddate') \
            .project('fielddate', 'board_id') \
            .join(
                boards.filter(nf.custom(lambda x: True if isinstance(x, list) and len(x) == 0 else False, 'board_permissions')),
                type='inner',
                by='board_id'
            ) \
            .groupby('fielddate') \
            .aggregate(uncollabs=na.count()) \
            .publish(self.get_statface_report('Collections/Metrics/collabs/uncollabs'), allow_change_job=True)
        yield 'collabs_uncollabs', collabs_uncollabs

    @require(
        'PrepBoards.collabs_and_potential',
        'CollectionsRedirLog.parsed',
        'Cards.parsed'
    )
    def card_additions(self, streams):
        collections_redir_log = streams['CollectionsRedirLog.parsed'].project(ne.all(exclude=('board_id', )))
        collabs = streams['PrepBoards.collabs_and_potential']
        cards = streams['Cards.parsed']
        dimensions = ('board_auto_approve_invite_requests', 'board_is_private', 'fielddate', 'is_owner')
        card_additions = collections_redir_log \
            .filter(nf.or_(nf.equals('path', 'finish.card.create'), nf.equals('path', 'finish.card.share'))) \
            .join(cards, by='card_id', type='inner', allow_undefined_keys=False) \
            .join(collabs, by_left='card_board_id', by_right='board_id', allow_undefined_keys=False) \
            .project('board_auto_approve_invite_requests', 'board_is_private', 'fielddate',
                     is_owner=ne.custom(lambda uid, owner: uid == owner, 'user_id', 'board_owner_id').with_type(bool)
                     ) \
            .cast(
                board_is_private=str,
                board_auto_approve_invite_requests=str,
                is_owner=str
            ) \
            .map(with_hints(output_schema={key: str for key in dimensions})(
                AddTotalsMapper(dimensions)
            )) \
            .groupby(*dimensions) \
            .aggregate(collab_card_count=na.count()) \
            .project('collab_card_count', *dimensions)
        yield 'card_additions', self.publish(
            card_additions, self.get_statface_report('Collections/Metrics/collabs/card_additions'), allow_change_job=True
        )
