# coding: utf-8

import sys
import datetime

from pytz import timezone

from analytics.plotter_lib.plotter import Plot, require
from analytics.plotter_lib.utils import get_element_by_path
from analytics.collections.plotter_collections.plots.utils import mongo_id_to_datestr
from analytics.collections.plotter_collections.plots.card_origin_metrics import get_card_origin_ui

from nile.api.v1 import (
    Record,
    aggregators as na,
    extractors as ne,
    filters as nf,
    with_hints,
    extended_schema,
)
from qb2.api.v1 import (
    typing as qt
)


class MapBoards(object):
    def __call__(self, records):
        for record in records:
            is_organic = int('moder' not in record['document'])
            yield Record(
                record,
                is_organic=is_organic,
            )


class BoardsParser(object):
    def __call__(self, recs):
        for rec in recs:
            try:
                outrec = {}

                outrec['board_id'] = rec.get('id')
                outrec['board_owner_id'] = rec.get('owner')
                outrec['board_slug'] = rec.document.get('slug')
                outrec['board_slugs'] = [outrec['board_slug']] + rec.document.get('old_slugs', [])

                created_at_str = get_element_by_path(rec.document, 'service.created_at._$date')
                dt_template = '%Y-%m-%dT%H:%M:%S.%fZ' if created_at_str.endswith('Z') else '%Y-%m-%dT%H:%M:%S'
                created_at_dt = datetime.datetime.strptime(created_at_str, dt_template)
                if created_at_dt > datetime.datetime.now():
                    continue
                outrec['board_created_at_datetime_utc'] = created_at_dt.strftime('%Y-%m-%dT%H:%M:%S.%fZ')
                outrec['board_created_at_date_utc'] = created_at_dt.strftime('%Y-%m-%d')
                created_at_dt = created_at_dt \
                    .replace(tzinfo=timezone('UTC')) \
                    .astimezone(timezone('Europe/Moscow'))
                outrec['board_created_at_datetime'] = created_at_dt.strftime('%Y-%m-%dT%H:%M:%S.%fZ')
                outrec['board_created_at_date'] = created_at_dt.strftime('%Y-%m-%d')
                outrec['board_created_at_timestamp'] = int(created_at_dt.strftime('%s'))
                outrec['board_title'] = rec.document.get('title')
                outrec['board_is_private'] = rec.document.get('is_private')
                outrec['board_is_default'] = rec.document.get('is_default')
                outrec['board_is_promo'] = rec.document.get('additional_attributes', {}).get('is_commercial', False)
                outrec['board_is_banned'] = rec.document.get('ban')
                outrec['board_moder_status'] = rec.document.get('moder', {}).get('status', 0)
                outrec['board_private_card_count'] = get_element_by_path(rec.document, 'stat.priv_cs')
                outrec['board_public_card_count'] = get_element_by_path(rec.document, 'stat.pub_cs')
                outrec['board_subscriber_count'] = get_element_by_path(rec.document, 'stat.subs')
                outrec['board_permissions'] = rec.document.get('permissions', [])
                outrec['board_description'] = rec.document.get('description')
                outrec['board_auto_approve_invite_requests'] = rec.document.get('auto_approve_invite_requests')

                corgie_v3 = get_element_by_path(rec.document, 'features.corgie_v3.value')
                if isinstance(corgie_v3, str):
                    outrec['board_features_corgie_v3'] = corgie_v3

                corgie_v2 = get_element_by_path(rec.document, 'features.corgie_v2.value')
                if isinstance(corgie_v2, str):
                    outrec['board_features_corgie_v2'] = corgie_v2
            except Exception as e:
                sys.stderr.write(str(e))
                continue
            yield Record.from_dict(outrec)


class CardsParser(object):
    def __call__(self, recs):
        for rec in recs:
            try:
                outrec = {}
                outrec['card_id'] = rec.get('id')
                outrec['card_board_id'] = rec.get('board')
                outrec['card_owner_id'] = rec.get('owner')
                outrec['card_source_type'] = get_element_by_path(rec.document, 'content.0.source_type')
                outrec['card_is_product'] = get_element_by_path(rec.document, 'meta_info.type', None) == 'product'

                created_at_str = get_element_by_path(rec.document, 'service.created_at._$date')
                dt_template = '%Y-%m-%dT%H:%M:%S.%fZ' if created_at_str.endswith('Z') else '%Y-%m-%dT%H:%M:%S'
                created_at_dt = datetime.datetime.strptime(created_at_str, dt_template)
                if created_at_dt > datetime.datetime.now():
                    continue
                outrec['card_created_at_datetime_utc'] = created_at_dt.strftime('%Y-%m-%dT%H:%M:%S.%fZ')
                outrec['card_created_at_date_utc'] = created_at_dt.strftime('%Y-%m-%d')

                created_at_dt = created_at_dt \
                    .replace(tzinfo=timezone('UTC')) \
                    .astimezone(timezone('Europe/Moscow'))

                outrec['card_created_at_datetime'] = created_at_dt.strftime('%Y-%m-%dT%H:%M:%S.%fZ')
                outrec['card_created_at_date'] = created_at_dt.strftime('%Y-%m-%d')
                outrec['card_created_at_timestamp'] = int(created_at_dt.strftime('%s'))
                outrec['card_description'] = rec.document.get('description')
                outrec['card_is_private'] = rec.document.get('is_private')
                outrec['card_is_banned'] = rec.document.get('ban')
                ban_mask = rec.document.get('ban_mask')
                outrec['card_ban_mask'] = ban_mask if isinstance(ban_mask, int) else None
                outrec['card_origin_client_name'] = get_element_by_path(rec.document, 'origin.client.name')
                outrec['card_origin_client_ui'] = get_element_by_path(rec.document, 'origin.client.ui')
                outrec['card_origin_action'] = get_element_by_path(rec.document, 'origin.action')
                outrec['card_origin_ui'] = get_card_origin_ui(outrec['card_origin_client_name'], outrec['card_origin_client_ui'])
                outrec['card_comment_count'] = get_element_by_path(rec.document, 'stats.comments_count')
                outrec['card_like_count'] = get_element_by_path(rec.document, 'stats.likes_count')
                outrec['card_share_count'] = get_element_by_path(rec.document, 'stats.shares_count')
                outrec['card_is_browser_bookmark'] = bool(rec.document.get('browser_bookmark_id', False))
                outrec['card_competitions'] = rec.document.get("competitions", [])
                outrec['card_height'] = get_element_by_path(rec.document, 'content.0.content.sizes.orig.height')
                outrec['card_width'] = get_element_by_path(rec.document, 'content.0.content.sizes.orig.width')
                outrec['card_cv_version'] = get_element_by_path(rec.document, 'content.0.content.avatars_meta.cv_version')
                outrec['card_orig_format'] = get_element_by_path(rec.document, 'content.0.content.avatars_meta.orig-format')
                outrec['card_good_quality'] = get_element_by_path(rec.document, 'content.0.content.avatars_meta.NeuralNetClasses.good_quality')
                outrec['card_page_domain'] = get_element_by_path(rec.document, 'source_meta.page_domain')
                outrec['card_avatars_url'] = 'https://avatars.mds.yandex.net/get-pdb/{}/{}/orig'.format(
                    str(get_element_by_path(rec.document, 'content.0.content.group_id')),
                    str(get_element_by_path(rec.document, 'content.0.content.avatars_key'))
                )
                outrec['card_meta_info_type'] = get_element_by_path(rec.document, 'meta_info.type')
                outrec['card_meta_info_data_title'] = get_element_by_path(rec.document, 'meta_info.data.title')
                outrec['card_url'] = 'https://yandex.ru/collections/card/{}/'.format(outrec['card_id'])
                outrec['card_md5'] = get_element_by_path(rec.document, 'content.0.content.avatars_meta.md5')
            except Exception as e:
                sys.stderr.write(str(e))
                continue
            yield Record.from_dict(outrec)


class TagTableParser(object):
    def __call__(self, recs):
        for rec in recs:
            outrec = {}
            outrec['board_id'] = rec.get('id')
            outrec['board_adv_prob'] = rec.extend_flat_tag_tree.get('obj.board.attr.inappropriate', {}).get('probability')

            yield Record.from_dict(outrec)


class UserParser(object):
    def __call__(self, recs):
        for rec in recs:
            try:
                outrec = {}
                outrec['user_id'] = rec.get('id')
                outrec['user_login'] = rec.document.get('login')
                outrec['user_puid'] = get_element_by_path(rec.document, 'uid._$numberLong')
                outrec['user_company_id'] = get_element_by_path(rec.document, 'company_id._$numberLong')
                outrec['user_companies'] = get_element_by_path(rec.document, 'companies')
                outrec['user_icookie'] = rec.document.get('yandexuid')
                outrec['user_name'] = rec.document.get('display_name')
                outrec['user_email'] = rec.document.get('email')
                outrec['user_is_banned'] = rec.document.get('ban')
                outrec['user_has_phone'] = rec.document.get('has_phones')
                outrec['user_labels'] = rec.document.get('labels', [])
                outrec['user_role'] = rec.document.get('role')
                created_date = mongo_id_to_datestr(outrec['user_id'])
                outrec['user_is_partner'] = 'verified' in rec.document.get('labels', []) and created_date >= '2019-10-01'
                outrec['user_created_date'] = created_date
                outrec['user_created_timestamp'] = int(mongo_id_to_datestr(outrec['user_id'], date_format='%s'))
                outrec['user_allow_chat'] = get_element_by_path(rec.document, 'prefs.general.allow_chat')
                outrec['user_ban'] = rec.document.get('ban')
                outrec['user_rubrics_names'] = []

                rubrics = get_element_by_path(rec.document, 'company_info.rubrics')
                if isinstance(rubrics, list):
                    for rubric in rubrics:
                        if not isinstance(rubric.get('names'), list):
                            continue

                        for rubric_name in rubric.get('names'):
                            if rubric_name.get('locale') == 'ru' and rubric_name.get('value'):
                                outrec['user_rubrics_names'].append(rubric_name.get('value'))
            except Exception as e:
                sys.stderr.write(str(e))
                continue
            yield Record.from_dict(outrec)


class Boards(Plot):
    @require('//home/collections-backups/db/prod/collections-fast-dump/latest/board')
    def get_all(self, streams):
        stream = streams['//home/collections-backups/db/prod/collections-fast-dump/latest/board']
        return stream.map(
            with_hints(
                output_schema=extended_schema(is_organic=int)
            )(
                MapBoards()
            )
        )

    @require('//home/collections-backups/db/prod/collections-fast-dump/latest/board')
    def parsed(self, streams):
        stream = streams['//home/collections-backups/db/prod/collections-fast-dump/latest/board']
        return stream.map(
            with_hints(
                output_schema=dict(
                    board_id=qt.Optional[qt.String],
                    board_owner_id=qt.Optional[qt.String],
                    board_created_at_datetime_utc=qt.Optional[qt.String],
                    board_created_at_date_utc=qt.Optional[qt.String],
                    board_created_at_datetime=qt.Optional[qt.String],
                    board_created_at_date=qt.Optional[qt.String],
                    board_created_at_timestamp=qt.Optional[qt.Int32],
                    board_title=qt.Optional[qt.String],
                    board_slug=qt.Optional[qt.String],
                    board_slugs=qt.Optional[qt.Yson],
                    board_is_private=qt.Optional[qt.Bool],
                    board_is_default=qt.Optional[qt.Bool],
                    board_is_banned=qt.Optional[qt.Bool],
                    board_moder_status=qt.Optional[qt.UInt32],
                    board_is_promo=qt.Optional[qt.Bool],
                    board_private_card_count=qt.Optional[qt.Integer],
                    board_public_card_count=qt.Optional[qt.Integer],
                    board_subscriber_count=qt.Optional[qt.Integer],
                    board_permissions=qt.Yson,
                    board_description=qt.Optional[qt.String],
                    board_auto_approve_invite_requests=qt.Optional[qt.Bool],
                    board_features_corgie_v2=qt.Optional[qt.String],
                    board_features_corgie_v3=qt.Optional[qt.String],
                )
            )(
                BoardsParser()
            )
        )


class Users(Plot):
    @require('Boards.get_all')
    def get_organics(self, streams):
        boards = streams['Boards.get_all']
        return boards \
            .groupby('owner') \
            .aggregate(is_organic=na.min('is_organic'))

    @require('//home/collections-backups/db/prod/collections-fast-dump/latest/user', '//home/collections/prod/ban_unban/dump')
    def parsed(self, streams):
        users = streams['//home/collections-backups/db/prod/collections-fast-dump/latest/user'] \
            .map(
                with_hints(
                    output_schema=dict(
                        user_id=qt.Optional[qt.String],
                        user_login=qt.Optional[qt.String],
                        user_puid=qt.Optional[qt.String],
                        user_company_id=qt.Optional[qt.String],
                        user_companies=qt.Optional[qt.Yson],
                        user_icookie=qt.Optional[qt.String],
                        user_name=qt.Optional[qt.String],
                        user_email=qt.Optional[qt.String],
                        user_is_banned=qt.Optional[qt.Bool],
                        user_is_partner=qt.Optional[qt.Bool],
                        user_created_date=qt.Optional[qt.String],
                        user_created_timestamp=qt.Optional[qt.Int32],
                        user_has_phone=qt.Optional[qt.Bool],
                        user_labels=qt.Yson,
                        user_role=qt.Optional[qt.String],
                        user_allow_chat=qt.Optional[qt.Bool],
                        user_ban=qt.Optional[qt.String],
                        user_rubrics_names=qt.Optional[qt.Yson]
                    )
                )(
                    UserParser()
                )
            )

        antispam_ban = streams['//home/collections/prod/ban_unban/dump'] \
            .filter(nf.equals('entity_type', 'user')) \
            .groupby('id') \
            .top(1, by='source_table_creation_time') \
            .filter(nf.equals('action_type', 'ban')) \
            .project(user_id='id', antispam_ban=ne.const(True))

        return users \
            .join(antispam_ban, by='user_id', type='left') \
            .project(
                ne.all(['user_ban', 'antispam_ban']),
                user_ban=ne.custom(lambda x, y: True if x or y else False, 'user_ban', 'antispam_ban').with_type(bool)
            )

    @require('Users.parsed')
    def partners(self, streams):
        return streams['Users.parsed'].filter(
            nf.equals('user_is_partner', True)
        )


class Cards(Plot):
    @require(
        '//home/collections-backups/db/prod/collections-fast-dump/latest/card',
        'Users.get_organics',
    )
    def get_organics(self, streams):
        cards = streams['//home/collections-backups/db/prod/collections-fast-dump/latest/card']
        users = streams['Users.get_organics']

        return cards \
            .join(users, type='left', by='owner') \
            .project(
                ne.all(),
                is_card_organic=ne.custom(lambda x: x if x is not None else 1, 'is_organic').with_type(int),
            )

    @require('//home/collections-backups/db/prod/collections-fast-dump/latest/card')
    def parsed(self, streams):
        stream = streams['//home/collections-backups/db/prod/collections-fast-dump/latest/card']

        return stream.map(
            with_hints(
                output_schema=dict(
                    card_id=qt.Optional[qt.String],
                    card_url=qt.Optional[qt.String],
                    card_md5=qt.Optional[qt.String],
                    card_board_id=qt.Optional[qt.String],
                    card_owner_id=qt.Optional[qt.String],
                    card_source_type=qt.Optional[qt.String],
                    card_created_at_datetime_utc=qt.Optional[qt.String],
                    card_created_at_date_utc=qt.Optional[qt.String],
                    card_created_at_datetime=qt.Optional[qt.String],
                    card_created_at_date=qt.Optional[qt.String],
                    card_created_at_timestamp=qt.Optional[qt.Int32],
                    card_description=qt.Optional[qt.String],
                    card_is_private=qt.Optional[qt.Bool],
                    card_is_banned=qt.Optional[qt.Bool],
                    card_is_product=qt.Optional[qt.Bool],
                    card_ban_mask=qt.Optional[qt.Integer],
                    card_origin_client_name=qt.Optional[qt.String],
                    card_origin_client_ui=qt.Optional[qt.String],
                    card_origin_ui=qt.Optional[qt.String],
                    card_origin_action=qt.Optional[qt.String],
                    card_comment_count=qt.Optional[qt.Integer],
                    card_like_count=qt.Optional[qt.Integer],
                    card_share_count=qt.Optional[qt.Integer],
                    card_is_browser_bookmark=qt.Optional[qt.Bool],
                    card_competitions=qt.Yson,
                    card_height=qt.Optional[qt.Integer],
                    card_width=qt.Optional[qt.Integer],
                    card_cv_version=qt.Optional[qt.String],
                    card_orig_format=qt.Optional[qt.String],
                    card_good_quality=qt.Optional[qt.Integer],
                    card_page_domain=qt.Optional[qt.String],
                    card_meta_info_type=qt.Optional[qt.String],
                    card_meta_info_data_title=qt.Optional[qt.String],
                    card_avatars_url=qt.Optional[qt.String],
                )
            )(
                CardsParser()
            )
        )


class TagTable(Plot):
    @require(
        'hahn://home/podborki/prod/tags/binarized_v2/board/full/latest',
        'arnold://home/collections/prod/tags/binarized_v2/board/full/latest',
    )
    def parsed(self, streams):
        return streams.values()[0].map(
            with_hints(
                output_schema=dict(
                    board_id=qt.Optional[qt.String],
                    board_adv_prob=qt.Optional[qt.Float]
                )
            )(
                TagTableParser()
            )
        )


class InvitationsBackup(Plot):
    @require('//home/collections/analytics/invitations')
    def get(self, streams):
        stream = streams['//home/collections/analytics/invitations']
        return stream.project(ne.all())


class AllowChatUsersBackup(Plot):
    @require('//home/collections/analytics/backups/chats/allow_chat_users')
    def get(self, streams):
        stream = streams['//home/collections/analytics/backups/chats/allow_chat_users']

        return stream.project(ne.all())


class CardChatsBackup(Plot):
    @require('//home/collections/analytics/backups/chats/card_chats')
    def get(self, streams):
        stream = streams['//home/collections/analytics/backups/chats/card_chats']

        return stream.project(ne.all())


class Channels(Plot):
    @require('//home/collections/analytics/channels')
    def get(self, streams):
        stream = streams['//home/collections/analytics/channels']
        return stream.project(ne.all())
