# coding: utf-8

from analytics.plotter_lib.plotter import Plot, require
from analytics.plotter_lib.utils import (
    date_range,
    split_by_dates,
    DATE_FORMAT
)
from utils import (
    get_card_id_from_url,
    get_user_login_board_slug_from_url
)

from nile.api.v1 import (
    Record,
    with_hints,
    extended_schema,
    extractors as ne,
    aggregators as na,
    filters as nf,
    grouping as ng
)
from qb2.api.v1 import (
    extractors as qe,
    filters as qf,
    typing as qt
)


def fix_reqids_reducer(groups):
    for key, records in groups:
        last_rec = None
        for r in records:
            if last_rec:
                if r.url == last_rec.url and r.req_id != last_rec.req_id:
                    r = r.transform(req_id=last_rec.req_id)
            last_rec = r

            yield r


def get_cost(event_cost, place_id, counter_type):
    if (place_id == 542 and counter_type == 2) or (place_id == 1542 and counter_type == 1):
        return event_cost * 30.0 / 1000000.0
    else:
        return 0


@with_hints(
    output_schema=dict(
        user_id=qt.Optional[qt.String],
        content_key=qt.Optional[qt.String],
        activation_date=qt.Optional[qt.String],
        activation_param=qt.Optional[qt.String],
        content_id=qt.Optional[qt.String],
        content_type=qt.Optional[qt.String],
        content_url=str
    ),
)
def user_with_content_key_reducer(groups):
    for key, records in groups:
        first_rec = None
        for r in records:
            if not first_rec:
                first_rec = r

                yield Record(
                    user_id=key.user_id,
                    content_key=first_rec.get('user_login'),
                    activation_date=first_rec.get('user_created_date'),
                    activation_param=first_rec.get('board_created_at_date'),
                    content_id=key.user_id,
                    content_type='profile',
                    content_url='https://yandex.ru/collections/user/{}'.format(first_rec.user_login)
                )
            if r.board_id:
                yield Record(
                    user_id=key.user_id,
                    content_key=r.board_id,
                    activation_date=first_rec.user_created_date,
                    activation_param=first_rec.board_created_at_date,
                    content_id=r.board_id,
                    content_type='board',
                    content_url='https://yandex.ru/collections/user/{}/{}'.format(first_rec.user_login, r.board_slug)
                )
                yield Record(
                    user_id=key.user_id,
                    content_key=r.user_login + r.board_slug,
                    activation_date=first_rec.user_created_date,
                    activation_param=first_rec.board_created_at_date,
                    content_id=r.board_id,
                    content_type='board',
                    content_url='https://yandex.ru/collections/user/{}/{}'.format(first_rec.user_login, r.board_slug)
                )
            if r.card_id:
                if first_rec:
                    yield Record(
                        user_id=key.user_id,
                        content_key=r.card_id,
                        activation_date=first_rec.user_created_date,
                        activation_param=first_rec.board_created_at_date,
                        content_id=r.card_id,
                        content_type='card',
                        content_url='https://yandex.ru/collections/card/{}/'.format(r.card_id)
                    )
                else:
                    yield Record(
                        user_id=key.user_id,
                        content_key=r.card_id,
                        activation_date=r.user_created_date,
                        activation_param=r.board_created_at_date,
                        content_id=r.card_id,
                        content_type='card',
                        content_url='https://yandex.ru/collections/card/{}/'.format(r.card_id)
                    )


def bad_id(content_id):
    if not isinstance(content_id, str):
        return True

    bad_ids = {'incut', 'user_recent', 'infinite', 'auth_likes', 'auth_cards', 'extension'}
    for bad_id in bad_ids:
        if bad_id in content_id:
            return True

    return False


@with_hints(
    output_schema=dict(fielddate=str, cost=float, content_key=str, joined_by=str, page_id=int, imp_id=int)
)
def extract_content_cost(groups):
    for key, records in groups:
        json_info_content_ids = set()
        card_id = None
        board_id = None
        board_slug = None
        login = None
        user_login = None

        for r in records:
            if r.path == 'access':
                if r.page_name == 'card' and r.card_id:
                    card_id = r.card_id
                    break
                elif r.page_name == 'board' and r.board_id:
                    board_id = r.board_id
                    break
            else:
                if isinstance(r.json_info, list):
                    for content in r.json_info:
                        if isinstance(content, dict) and content.get('id') and not bad_id(content.get('id')):
                            json_info_content_ids.add(content['id'])
                elif r.url:
                    if not card_id:
                        card_id = get_card_id_from_url(r.url)
                    if not board_slug:
                        login, board_slug = get_user_login_board_slug_from_url(r.url)
                    if login:
                        user_login = login

        if card_id:
            yield Record(
                fielddate=key.fielddate,
                cost=key.cost,
                page_id=key.page_id,
                imp_id=key.imp_id,
                content_key=card_id,
                joined_by='card_id'
            )
        elif board_id:
            yield Record(
                fielddate=key.fielddate,
                cost=key.cost,
                page_id=key.page_id,
                imp_id=key.imp_id,
                content_key=board_id,
                joined_by='board_id'
            )
        elif json_info_content_ids:
            for content_id in json_info_content_ids:
                if content_id:
                    yield Record(
                        fielddate=key.fielddate,
                        cost=float(key.cost) / len(json_info_content_ids),
                        page_id=key.page_id,
                        imp_id=key.imp_id,
                        content_key=content_id,
                        joined_by='json_info_content'
                    )
        elif board_slug and user_login:
            yield Record(
                fielddate=key.fielddate,
                cost=key.cost,
                page_id=key.page_id,
                imp_id=key.imp_id,
                content_key=user_login + board_slug,
                joined_by='user_login_board_slug'
            )
        elif user_login:
            yield Record(
                fielddate=key.fielddate,
                cost=key.cost,
                page_id=key.page_id,
                imp_id=key.imp_id,
                content_key=user_login,
                joined_by='user_login'
            )
        else:
            yield Record(
                fielddate=key.fielddate,
                cost=key.cost,
                page_id=key.page_id,
                imp_id=key.imp_id,
                content_key=r.url,
                joined_by='url'
            )


class CollectionsMoney(Plot):
    @require('//home/yabs/dict/Page')
    def get_collections_pageids(self, streams):
        return streams['//home/yabs/dict/Page'] \
            .filter(nf.custom(lambda name: 'collections.yandex' in name, 'Name')) \
            .project(page_id='PageID') \
            .unique('page_id')

    @require('chevent-log')
    def get_raw_chevent_log(self):
        return {}

    @require('CollectionsMoney.get_collections_pageids', 'CollectionsMoney.get_raw_chevent_log')
    def get_collections_chevent_log_with_reqids(self, streams):
        return streams['CollectionsMoney.get_raw_chevent_log'] \
            .qb2(
            log='bs-chevent-cooked-log',
            fields=[
                'page_id',
                'reqid',
                'event_cost',
                'place_id',
                'counter_type',
                qe.log_field('impid').with_type(int),
                qe.custom(
                    'cost',
                    get_cost,
                    'event_cost',
                    'place_id',
                    'counter_type'
                ).with_type(float),
            ]
        ) \
            .project(ne.all(['reqid', 'impid']), req_id='reqid', imp_id='impid') \
            .join(streams['CollectionsMoney.get_collections_pageids'], type='inner', by='page_id')

    @require('CollectionsRedirLog.parsed')
    def fix_redir_reqids(self, streams):
        return streams['CollectionsRedirLog.parsed'] \
            .groupby('fielddate', 'icookie') \
            .sort('timestamp', 'req_id') \
            .groupby(ng.sessions()) \
            .reduce(with_hints(output_schema=extended_schema())(fix_reqids_reducer))

    @require('CollectionsMoney.fix_redir_reqids', 'CollectionsMoney.get_collections_chevent_log_with_reqids')
    def joined_redir_with_cost(self, streams):
        return streams['CollectionsMoney.get_collections_chevent_log_with_reqids'] \
            .groupby('req_id') \
            .aggregate(
                cost=na.sum('cost'),
                page_id=na.any('page_id'),
                imp_id=na.any('imp_id')
            ) \
            .filter(nf.not_(nf.equals('cost', 0))) \
            .join(streams['CollectionsMoney.fix_redir_reqids'], by='req_id', type='inner')

    @require('CollectionsMoney.joined_redir_with_cost', 'PrepCards.with_organic_flag', 'PrepBoards.with_organic_flag')
    def get_authors_money(self, streams):
        user_with_content_key = self.job \
            .concat(
                streams['PrepBoards.with_organic_flag'].project(
                    'user_id',
                    'user_login',
                    'user_created_date',
                    'board_id',
                    'board_slug',
                    'board_created_at_date',
                    sort_p=ne.custom(lambda x: int(x), 'board_is_private').with_type(int)
                ),
                streams['PrepCards.with_organic_flag'].project(
                    'user_id',
                    'user_created_date',
                    'card_id',
                    sort_p=ne.const(2)
                )
            ) \
            .filter(qf.nonzero('user_id')) \
            .groupby('user_id') \
            .sort('sort_p', 'board_created_at_date') \
            .reduce(user_with_content_key_reducer)

        return streams['CollectionsMoney.joined_redir_with_cost'] \
            .groupby('fielddate', 'req_id', 'page_id', 'imp_id', 'cost') \
            .reduce(extract_content_cost) \
            .join(user_with_content_key, by='content_key', type='left') \
            .project(
                ne.all('content_id'),
                content_id=ne.custom(lambda x, y: x if x else y, 'content_id', 'content_key').with_type(qt.Optional[qt.String])) \
            .groupby('fielddate', 'user_id', 'content_id', 'page_id', 'imp_id') \
            .aggregate(
                cost=na.sum('cost'),
                activation_date=na.any('activation_date'),
                activation_param=na.any('activation_param'),
                content_type=na.any('content_type'),
                content_url=na.any('content_url'),
                joined_by=na.any('joined_by')
            ) \
            .project(
                'fielddate',
                'user_id',
                'content_id',
                'page_id',
                'imp_id',
                'cost',
                activation_date=ne.custom(lambda x: str(x), 'activation_date').with_type(str),
                activation_param=ne.custom(lambda x: str(x), 'activation_param').with_type(str),
                content_type=ne.custom(lambda x: str(x), 'content_type').with_type(str),
                content_url=ne.custom(lambda x: str(x), 'content_url').with_type(str),
                joined_by=ne.custom(lambda x: str(x), 'joined_by').with_type(str)
            )
        # TODO: убрать project после https://st.yandex-team.ru/YQL-9651

    @require('CollectionsMoney.get_authors_money')
    def put_authors_money(self, streams):
        for table_stream in split_by_dates(
                streams['CollectionsMoney.get_authors_money'],
                [x.strftime(DATE_FORMAT) for x in date_range(self.date, self.dateend)],
                '//home/collections/analytics/backups/authors/money/',
                sort_by=['cost']
        ):
            yield table_stream
