# coding: utf-8

from copy import deepcopy
from urllib import unquote

from qb2.api.v1 import (
    typing as qt,
    filters as qf,
)
from nile.api.v1 import (
    Record,
    with_hints,
    extended_schema,
    filters as nf,
    extractors as ne,
    aggregators as na,
)

from analytics.plotter_lib.plotter import Plot, require
from analytics.collections.plotter_collections.plots.utils import get_card_id_from_url, get_user_login_board_slug_from_url

COLLECTIONS_SHOWS_SQUEEZE_DEFAULT_SCHEMA = {
    'service': qt.String,
    'page_name': qt.Optional[qt.String],
    'location': qt.Optional[qt.String],
    'ui': qt.Optional[qt.String],
    'timestamp': qt.Int32,
    'fielddate': qt.String,
    'reqid': qt.Optional[qt.String],
    'yandexuid': qt.Optional[qt.String],
    'puid': qt.Optional[qt.String],
    'user_id': qt.Optional[qt.String],
    'content_type': qt.String,
    'content_subtype': qt.Optional[qt.String],
    'content_position': qt.Optional[qt.Int32],
    'card_id': qt.Optional[qt.String],
    'board_id': qt.Optional[qt.String],
}


def get_reqid(record):
    """Для Коллекций в качестве reqid используем какой-нибудь имеющийся id сессии"""
    for key in ['feed_session_id', 'cards_session_id', 'boards_session_id', 'card_boards_session_id', 'req_id']:
        if record.get(key):
            return record.get(key)


def extract_track_objects_item_data(item):
    INVALID_TYPES = ('advCards', 'userCards', 'user', 'channelRelated', 'favoritesCards', 'editorsChoice')
    BOARDS_WHITELIST = ('feedRecommendations',)
    BOARDS_BLACKLIST = ('onboard',)

    item_id = str(item.get('id'))
    if item_id.startswith('footer/'):
        item_id = item_id[7:]
    if len(item_id) != 24:
        return

    content_subtype = str(item.get('type')) if 'type' in item else None
    if content_subtype in INVALID_TYPES:
        return

    content_data = {}
    is_content_board = False
    if content_subtype:
        if content_subtype.startswith('cards$cardSimilarToCard/') or content_subtype.startswith('cards$similar/'):
            parts = content_subtype.split('/')
            content_subtype = parts[0]
        content_data['content_subtype'] = content_subtype

        # борды определяем по белому списку как BoardSchema в https://github.yandex-team.ru/mm-interfaces/fiji/blob/dev/collections/src/schema/state/State.ts
        # https://github.yandex-team.ru/mm-interfaces/fiji/blob/dev/collections/src/entities/Shaker/Shaker.ts#L6
        if content_subtype in BOARDS_WHITELIST:
            is_content_board = True
        elif ('board' in content_subtype.lower() and 'card' not in content_subtype.lower()) and content_subtype not in BOARDS_BLACKLIST:
            is_content_board = True

    if is_content_board:
        content_data['content_type'] = 'board'
        content_data['board_id'] = item_id
    else:
        # во всех остальных случаях — считаем что показана карточка
        content_data['content_type'] = 'card'
        content_data['card_id'] = item_id

    if 'pos' in item:
        content_data['content_position'] = int(item.get('pos'))

    return content_data


def get_redir_log_squeeze_base_data(record):
    return {
        'service': 'collections',
        'page_name': record.page_name,
        'location': record.loc,
        'ui': record.ui,
        'timestamp': int(record.timestamp),
        'fielddate': record.fielddate,
        'yandexuid': ('y' + record.yandexuid) if record.yandexuid else None,
        'puid': ('p' + record.puid) if record.puid else None,
        'user_id': record.user_id,
        'reqid': get_reqid(record),
    }


@with_hints(output_schema=COLLECTIONS_SHOWS_SQUEEZE_DEFAULT_SCHEMA)
def extract_card_page_shows(records):
    for r in records:
        card_id = get_card_id_from_url(r.url)
        if not card_id:
            continue

        content_data = get_redir_log_squeeze_base_data(r)
        content_data['location'] = 'page'
        content_data['content_type'] = 'card'
        content_data['card_id'] = card_id
        yield Record.from_dict(content_data)


@with_hints(output_schema=COLLECTIONS_SHOWS_SQUEEZE_DEFAULT_SCHEMA)
def extract_board_page_shows(records):
    for r in records:
        board_id = r.board_id
        if not board_id or len(board_id) != 24:
            continue

        content_data = get_redir_log_squeeze_base_data(r)
        content_data['location'] = 'page'
        content_data['content_type'] = 'board'
        content_data['board_id'] = board_id
        yield Record.from_dict(content_data)


@with_hints(output_schema=dict(
    COLLECTIONS_SHOWS_SQUEEZE_DEFAULT_SCHEMA,
    **{'user_login': qt.Optional[qt.String]}
))
def extract_profile_page_shows(records):
    for r in records:
        user_login, board_slug = get_user_login_board_slug_from_url(r.url)
        if not user_login:
            continue

        content_data = get_redir_log_squeeze_base_data(r)
        content_data['location'] = 'page'
        content_data['content_type'] = 'user'
        content_data['user_login'] = user_login
        yield Record.from_dict(content_data)


@with_hints(output_schema=dict(card_id=str))
def extract_cards_link_external_ids(records):
    for r in records:
        if r.get('card_id'):
            yield Record(
                card_id=r.card_id,
            )
            continue

        card_id = get_card_id_from_url(r.url)
        if card_id:
            yield Record(
                card_id=card_id,
            )
            continue


@with_hints(output_schema=dict(
    user_login=str,
    board_slug=str,
))
def extract_boards_link_external_ids(records):
    for r in records:
        user_login, board_slug = get_user_login_board_slug_from_url(r.url)
        if user_login and board_slug:
            yield Record(
                user_login=unquote(user_login),
                board_slug=unquote(board_slug),
            )


@with_hints(output_schema=extended_schema())
def extract_first_greenurls(groups):
    """Приджоинивает greenurls к первой записи без колонки greenurls, возвращает остальной поток записей как есть"""
    for key, records in groups:
        greenurls = None
        main_records = []
        for r in records:
            if r.get('greenurls'):
                greenurls = r.get('greenurls')
            else:
                main_records.append(r)

        for r in main_records:
            if greenurls:
                yield Record(r, greenurls=greenurls)
                greenurls = None
            else:
                yield r


@with_hints(output_schema=COLLECTIONS_SHOWS_SQUEEZE_DEFAULT_SCHEMA)
def extract_track_objects_shows(records):
    for r in records:
        base_data = get_redir_log_squeeze_base_data(r)
        for item in r.get('json_info', []):
            if item.get('finish') > item.get('start'):
                content_data = deepcopy(base_data)
                current_content_data = extract_track_objects_item_data(item)
                if current_content_data:
                    content_data.update(current_content_data)
                    yield Record.from_dict(content_data)


class CollectionsShows(Plot):

    @require('CollectionsRedirLog.parsed')
    def parse_track_objects_shows(self, streams):
        return streams['CollectionsRedirLog.parsed'] \
            .filter(
                qf.defined('json_info'),
            ) \
            .map(extract_track_objects_shows)

    @require('CollectionsRedirLog.parsed')
    def parse_card_page_shows(self, streams):
        return streams['CollectionsRedirLog.parsed'] \
            .filter(
                nf.and_(
                    nf.equals('page_name', 'card'),
                    nf.equals('path', 'access'),
                )
            ) \
            .map(extract_card_page_shows)

    @require('CollectionsRedirLog.parsed')
    def parse_board_page_shows(self, streams):
        return streams['CollectionsRedirLog.parsed'] \
            .filter(
                nf.and_(
                    nf.equals('page_name', 'board'),
                    nf.equals('path', 'access'),
                )
            ) \
            .map(extract_board_page_shows)

    @require('CollectionsRedirLog.parsed')
    def parse_profile_page_shows(self, streams):
        return streams['CollectionsRedirLog.parsed'] \
            .filter(
                nf.and_(
                    nf.equals('page_name', 'profile'),
                    nf.equals('path', 'access'),
                )
            ) \
            .map(extract_profile_page_shows)

    @require('CollectionsRedirLog.parsed')
    def parse_cards_link_external_counters(self, streams):
        """Парсит счётчики переходов на сайты с карточек
        Возвращает: card_id, greenurls"""
        return streams['CollectionsRedirLog.parsed'] \
            .filter(nf.equals('path', 'click.link_external')) \
            .map(extract_cards_link_external_ids) \
            .groupby('card_id') \
            .aggregate(
                greenurls=na.count()
            )

    @require(
        'CollectionsRedirLog.parsed',
        'PrepBoards.expanded_slugs',
    )
    def parse_boards_link_external_counters(self, streams):
        """Парсит счётчики переходов на сайты со страниц бордов
        Возвращает: board_id, greenurls"""
        return streams['CollectionsRedirLog.parsed'] \
            .filter(nf.equals('path', 'click.link_external')) \
            .map(extract_boards_link_external_ids) \
            .groupby('user_login', 'board_slug') \
            .aggregate(
                greenurls=na.count()
            ) \
            .join(
                streams['PrepBoards.expanded_slugs'].project('user_login', 'board_slug', 'board_id'),
                type='inner',
                by=['user_login', 'board_slug']
            ) \
            .groupby('board_id') \
            .aggregate(
                # суммируем статистику для бордов с разными slug'ами, но одинаковыми board_id'ами
                greenurls=na.sum('greenurls')
            )

    @require(
        'CollectionsShows.parse_track_objects_shows',
        'CollectionsShows.parse_card_page_shows',
        'CollectionsShows.parse_board_page_shows',
        'CollectionsShows.parse_cards_link_external_counters',
        'CollectionsShows.parse_boards_link_external_counters',
    )
    def all_cards_boards(self, streams):
        """К выжимкам показов карточек/бордов приклеивает переходы по гринурлам - колонка greenurls"""
        cards_greenurls = streams['CollectionsShows.parse_cards_link_external_counters'] \
            .project('card_id', 'greenurls', content_type=ne.const('card').with_type(str))

        boards_greenurls = streams['CollectionsShows.parse_boards_link_external_counters'] \
            .project('board_id', 'greenurls', content_type=ne.const('board').with_type(str))

        return self.job.concat(
            streams['CollectionsShows.parse_track_objects_shows'],
            streams['CollectionsShows.parse_card_page_shows'],
            streams['CollectionsShows.parse_board_page_shows'],
            ) \
            .concat(cards_greenurls, boards_greenurls) \
            .groupby('content_type', 'card_id', 'board_id') \
            .reduce(
                extract_first_greenurls,
                memory_limit=16*1024,
            )

    @require(
        'CollectionsShows.parse_profile_page_shows',
        'PrepUsers.with_organic_flag',
    )
    def users_shows_joined_with_db(self, streams):
        shows = streams['CollectionsShows.parse_profile_page_shows']
        users = streams['PrepUsers.with_organic_flag'].project(
            'user_login',
            content_owner_id='user_id',
            is_owner_organic='user_is_organic',
            is_owner_partner='user_is_partner',
            content_created_date='user_created_date',
            content_created_timestamp='user_created_timestamp',
            content_owner_puid=ne.custom(lambda x: None if x is None else ('p' + x), 'user_puid').with_type(qt.Optional[qt.String]),
            content_owner_company_id=ne.custom(lambda x: None if x is None else x, 'user_company_id').with_type(qt.Optional[qt.String]),
        )
        return shows \
            .join(users, type='inner', by='user_login')
