# coding: utf-8

import itertools

from analytics.plotter_lib.plotter import Plot, require
from nile.api.v1 import (
    Record,
    with_hints,
    extended_schema,
    extractors as ne,
    aggregators as na,
    filters as nf
)


@with_hints(output_schema=extended_schema())
def add_totals(recs):
    for r in recs:
        for comb in itertools.product(
                (r.action_type, '_total_'),
                (r.loc, '_total_'),
                (r.success, '_total_'),
                (r.device, '_total_')
        ):
            action_type, loc, success, device = comb
            yield dict(
                fielddate=r.fielddate,
                action_type=action_type,
                loc=loc,
                success=success,
                device=device,
            )


@with_hints(output_schema=extended_schema(action_count=int))
def precompute(recs):
    outdict = {}
    for r in recs:
        outdict[(r.fielddate, r.action_type, r.loc, r.success, r.device)] = outdict.get((r.fielddate, r.action_type, r.loc, r.success, r.device), 0) + 1

    for k, v in outdict.items():
        yield Record(fielddate=k[0], action_type=k[1], loc=k[2], success=k[3], device=k[4], action_count=v)


class CommentatorCommonMetrics(Plot):
    @require('CmntAPIAccessLog.get_collections')
    def calculate_commentator_report(self, streams):
        commentator = streams['CmntAPIAccessLog.get_collections']

        return commentator.filter(nf.equals('Success', True)) \
            .project(
                'fielddate',
                action_type=ne.custom(lambda x: str(x) if isinstance(x, str) else 'empty', 'UserAction').with_type(str),
                loc=ne.custom(lambda x: str(x) if isinstance(x, str) else 'empty', 'Stats_loc').with_type(str),
                success=ne.custom(lambda x: str(x) if isinstance(x, bool) else 'empty', 'Success').with_type(str),
                device=ne.custom(lambda x: str(x) if isinstance(x, str) else 'empty', 'device').with_type(str)
            ) \
            .map(add_totals) \
            .sort('fielddate', 'action_type', 'loc', 'success', 'device') \
            .map(precompute) \
            .groupby('fielddate', 'action_type', 'loc', 'success', 'device') \
            .aggregate(action_count=na.sum('action_count')) \
            .publish(self.get_statface_report('Collections/Metrics/social/CommentatorCommonReportV1'), allow_change_job=True)
