# coding: utf-8


from analytics.plotter_lib.plotter import Plot, require
from analytics.plotter_lib.utils import AddTotalsMapper
from nile.api.v1 import (
    Record,
    with_hints,
    aggregators as na,
    filters as nf,
    extractors as ne,
    extended_schema
)


from qb2.api.v1 import (
    typing as qt
)


@with_hints(
    output_schema=dict(
        fielddate=qt.Optional[qt.String],
        path=qt.Optional[qt.String],
        card_id=qt.Optional[qt.String],
        board_id=qt.Optional[qt.String],
        reason=qt.Optional[qt.String]
    )
)
def filter_complains(recs):
    for rec in recs:
        if rec.path and 'complain' in rec.path and rec.reason:
            if rec.card_id and '_url' in rec.card_id:
                yield Record(fielddate=rec.fielddate, path='click.card.complain.external', card_id=rec.card_id, reason=rec.reason)
            else:
                yield Record(fielddate=rec.fielddate, path=rec.path, card_id=rec.card_id, reason=rec.reason)


class Complains(Plot):
    @require(
        'CollectionsRedirLog.parsed',
        'Cards.parsed',
        'Boards.parsed'
    )
    def calculate(self, streams):
        collections_redir_log = streams['CollectionsRedirLog.parsed']
        cards = streams['Cards.parsed']
        boards = streams['Boards.parsed']

        collections_redir_log = collections_redir_log \
            .project('board_id', 'card_id', 'path', 'reason', 'fielddate') \
            .map(filter_complains)

        banned_cards = cards \
            .filter(nf.equals('card_is_banned', True)) \
            .project('card_id', 'card_is_banned')
        banned_boards = boards \
            .filter(nf.equals('board_is_banned', True)) \
            .project('board_id', 'board_is_banned')

        return collections_redir_log \
            .join(banned_cards, by='card_id', type='left') \
            .join(banned_boards, by='board_id', type='left') \
            .project(
                'fielddate',
                'path',
                'reason',
                is_banned=ne.custom(lambda x, y: True if x or y else False, 'card_is_banned', 'board_is_banned').with_type(bool)
            ) \
            .map(
                with_hints(output_schema=extended_schema())(
                    AddTotalsMapper(['fielddate', 'path', 'reason', 'is_banned'], ['fielddate', 'is_banned'])
                )
            ) \
            .groupby('fielddate', 'path', 'reason') \
            .aggregate(
                click_complain_count=na.count(),
                ban_count=na.count(nf.equals('is_banned', True))
            ) \
            .publish(self.get_statface_report('Collections/Metrics/bans/redir_complains_and_bans'), allow_change_job=True)
