# coding: utf-8

import datetime

from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from analytics.plotter_lib.utils import calc_retention_from_action
from nile.api.v1 import (
    with_hints,
    extractors as ne,
    aggregators as na,
    filters as nf,
    Record
)

RETENTION_PERIODS = (7, 14, 30)


@with_hints(output_schema=dict(fielddate=str, period=int, users=int, returned=int))
def corgie_authors_retention_reducer(groups):
    for key, recs in groups:

        corgie_dates = []
        card_dates = []
        for rec in recs:
            if rec.is_corgie:
                corgie_dates.append(rec.fielddate)
            else:
                card_dates.append(rec.fielddate)

        corgie_dates.sort()
        card_dates.sort()

        good_dates = set(calc_retention_from_action(corgie_dates, card_dates))
        corgie_dates = set(corgie_dates)

        for corgie_date in corgie_dates:
            for period in RETENTION_PERIODS:
                yield Record(
                    fielddate=corgie_date,
                    period=period,
                    users=1,
                    returned=((corgie_date, period) in good_dates)
                )


@with_hints(output_schema=dict(fielddate=str, period=int, users=int, returned=int))
def pub_card_retention_reducer(groups):
    for key, recs in groups:

        card_dates = []
        for rec in recs:
            card_dates.append(rec.fielddate)

        card_dates.sort()

        good_dates = set(calc_retention_from_action(card_dates, card_dates))
        card_dates = set(card_dates)

        for card_date in card_dates:
            for period in RETENTION_PERIODS:
                yield Record(
                    fielddate=card_date,
                    period=period,
                    users=1,
                    returned=((card_date, period) in good_dates)
                )


class CorgieAuthorsRetention(Plot):
    @require('Corgie.corgie', 'PrepCards.organic_cards')
    def corgie_authors_retention_publish(self, streams):
        cards = streams['PrepCards.organic_cards'].filter(nf.equals('card_is_private', False)) \
            .project(owner_id='card_owner_id', fielddate='card_created_at_date', is_corgie=ne.const(0)) \
            .unique('fielddate', 'owner_id') \
            .checkpoint(self.get_checkpoint_name('cards'))

        lower_bound_date = (self.dateend - datetime.timedelta(days=self.additional_days)).strftime(DATE_FORMAT)
        corgie = streams['Corgie.corgie'] \
            .filter(nf.custom(lambda x: x >= lower_bound_date, 'good_board_fielddate')) \
            .project(fielddate='good_board_fielddate', owner_id='board_owner_id', is_corgie=ne.const(1)) \
            .unique('fielddate', 'owner_id') \
            .checkpoint(self.get_checkpoint_name('corgie')) \

        corgie.concat(cards) \
            .groupby('owner_id') \
            .sort('fielddate', 'is_corgie') \
            .reduce(corgie_authors_retention_reducer, memory_limit=2*1024) \
            .groupby('fielddate', 'period') \
            .aggregate(
                returned=na.sum('returned'),
                users=na.sum('users')
            ) \
            .project(
                ne.all(),
                retention_type=ne.const('new_card'),
                returned_rate=ne.custom(lambda x, y: float(x) / y, 'returned', 'users').with_type(float)
            ) \
            .publish(self.get_statface_report('Collections/Metrics/Retention/RetentionInAction'), allow_change_job=True)

    @require('Corgie.corgie', 'PrepCards.organic_cards_without_spam')
    def pub_card_retention_publish(self, streams):
        lower_bound_date = (self.dateend - datetime.timedelta(days=self.additional_days)).strftime(DATE_FORMAT)
        cards = streams['PrepCards.organic_cards_without_spam'] \
            .filter(
                nf.and_(
                    nf.equals('card_is_private', False),
                    nf.custom(lambda x: x >= lower_bound_date, 'card_created_at_date')
                )
            ) \
            .project(owner_id='card_owner_id', fielddate='card_created_at_date', is_corgie=ne.const(0)) \
            .unique('fielddate', 'owner_id') \
            .checkpoint(self.get_checkpoint_name('pub_to_pub_cards'))

        return cards.groupby('owner_id') \
            .sort('fielddate') \
            .reduce(pub_card_retention_reducer, memory_limit=2*1024) \
            .groupby('fielddate', 'period') \
            .aggregate(
                returned=na.sum('returned'),
                users=na.sum('users')
            ) \
            .project(
                ne.all(),
                retention_type=ne.const('pub_to_pub_card'),
                returned_rate=ne.custom(lambda x, y: float(x) / y, 'returned', 'users').with_type(float)
            ) \
            .publish(self.get_statface_report('Collections/Metrics/Retention/RetentionInAction'), allow_change_job=True)
