# coding: utf-8

from analytics.plotter_lib.utils import AddTotalsMapper
from analytics.plotter_lib.plotter import Plot, require
from analytics.collections.plotter_collections.plots.utils import is_useful_event

from nile.api.v1 import (
    with_hints,
    extended_schema,
    extractors as ne,
    aggregators as na,
    filters as nf,
    Record,
    grouping
)


def convert_ui(ui):
    if ui in ('yandexApp', 'mobile'):
        return 'mobile'
    elif ui in ('tablet', 'desktop'):
        return 'desktop'
    elif not ui:
        return 'undefined'
    return ui


@with_hints(output_schema=extended_schema(
    sessions=int,
    useful_sessions=int,
    first_useful_event=str,
))
def useful_sessions_reducer(groups):
    for key, session in groups:
        first_access = None
        useful_event_path = 'empty'
        for event in session:
            if not first_access and event.path == 'access':
                first_access = event
            elif first_access and useful_event_path == 'empty':
                if is_useful_event(event):
                    useful_event_path = event.path
                    break

        if first_access:
            yield Record(
                fielddate=key.fielddate,
                yandexuid=key.yandexuid,
                ui=first_access.ui,
                page_name=first_access.page_name,
                sessions=1,
                useful_sessions=useful_event_path != 'empty',
                first_useful_event=useful_event_path
            )


@with_hints(output_schema=extended_schema())
def add_full_sessions(groups):
    for key, recs in groups:
        sessions = 0
        for rec in recs:
            if not sessions:
                sessions = rec.sessions
            outrec = rec.to_dict()

            outrec['sessions'] = sessions

            yield outrec


class EntryPointsUsefulSessions(Plot):
    @require(
        'CmntAPIAccessLog.convert_to_redir_format',
        'CollectionsRedirLog.parsed'
    )
    def get_useful_sessions(self, streams):
        cmnt_log = streams['CmntAPIAccessLog.convert_to_redir_format'].project(ne.all('iso_eventtime'))

        return streams['CollectionsRedirLog.parsed'] \
            .project('fielddate', 'timestamp', 'cts', 'yandexuid', 'path', 'ui', 'page_name') \
            .concat(cmnt_log) \
            .project(
                ne.all(['ui']),
                ui=ne.custom(convert_ui, 'ui').with_type(str),
            ) \
            .groupby('yandexuid', 'fielddate') \
            .sort('timestamp', 'cts') \
            .groupby(grouping.sessions()) \
            .reduce(useful_sessions_reducer, memory_limit=16*1024) \
            .filter(nf.custom(lambda x: True if x else False, 'page_name'))

    @require('EntryPointsUsefulSessions.get_useful_sessions')
    def entry_points_useful_sessions_publish(self, streams):
        add_totals_key_fields = ['fielddate', 'ui', 'page_name', 'sessions', 'useful_sessions', 'first_useful_event']
        add_totals_pass_fields = ['fielddate', 'sessions', 'useful_sessions']

        return streams['EntryPointsUsefulSessions.get_useful_sessions'] \
            .map(
                with_hints(output_schema=extended_schema())(
                    AddTotalsMapper(add_totals_key_fields, add_totals_pass_fields)
                )
            ) \
            .groupby('fielddate', 'ui', 'page_name', 'first_useful_event') \
            .aggregate(
                sessions=na.sum('sessions'),
                useful_sessions=na.sum('useful_sessions'),
            ) \
            .project(ne.all(), sort_p=ne.custom(lambda x: -x, 'sessions').with_type(int)) \
            .groupby('fielddate', 'ui', 'page_name') \
            .sort('sort_p') \
            .reduce(add_full_sessions) \
            .project(ne.all('sort_p'), useful_rate=ne.custom(lambda x, y: float(x) / y, 'useful_sessions', 'sessions').with_type(float)) \
            .publish(self.get_statface_report('Collections/Metrics/social/EntryPointUsefulSessions'), allow_change_job=True)
