# coding: utf-8

import re

from analytics.plotter_lib.plotter import Plot, require
from analytics.collections.plotter_collections.plots.utils import is_useful_event

from nile.api.v1 import (
    with_hints,
    aggregators as na,
    filters as nf,
    Record,
    grouping,
)
from qb2.api.v1 import (
    typing as qt,
)


@with_hints(output_schema=dict(
    fielddate=str,
    ui=str,
    yandexuid=str,
    timestamp=int,
    path=str,
    page_name=qt.Optional[qt.String],
    url=str,
    traffic_source=str,
    user_login=qt.Optional[qt.String],
    board_slug=qt.Optional[qt.String],
    useful_events=int,
))
def informer_sessions_reducer(groups):
    """Считает количество полезных событий после перехода на борду по информеру на морде"""
    def get_result_record(access, events_count):
        user_login, board_slug = None, None
        matches = re.match(r'https://yandex\..+?/collections/user/(.+?)/(.+?)/', access.url)
        if matches:
            user_login = matches.groups()[0]
            board_slug = matches.groups()[1]

        return Record(
            fielddate=access.fielddate,
            ui=access.ui,
            yandexuid=access.yandexuid,
            timestamp=access.timestamp,
            path=access.path,
            page_name=access.page_name,
            url=access.url,
            traffic_source=access.traffic_source,

            user_login=user_login,
            board_slug=board_slug,
            useful_events=events_count,
        )

    for key, session in groups:
        first_access = None
        useful_events = 0
        for event in session:
            if event.path == 'start.session' and event.traffic_source == 'morda_informer':
                if first_access is not None:
                    yield get_result_record(first_access, useful_events)
                    useful_events = 0
                first_access = event

            if first_access is not None and is_useful_event(event):
                useful_events += 1

        if first_access is not None:
            yield get_result_record(first_access, useful_events)


@with_hints(output_schema=dict(fielddate=str, metric=str, value=int))
def expand_metrics(records):
    for r in records:
        fielddate = r.fielddate
        record_dict = r.to_frozen_dict()

        for key in record_dict.keys():
            if key != 'fielddate':
                value = record_dict[key] if record_dict[key] is not None else 0
                yield Record(
                    fielddate=fielddate,
                    metric=key,
                    value=value,
                )


class InformerUsefulSessions(Plot):
    @require(
        'CollectionsRedirLog.parsed',
        'PrepUsers.with_organic_flag',
    )
    def boards(self, streams):
        """Возвращает статистику по бордам — сколько было полезных действий и переходов с информера
        считает также:
            * количество органических коллекций
            * количество партнёрских коллекций
            * TODO: джоин с бордами по slugs_history для:
                * количество свежих коллекций
                * количество коржей
        """
        users = streams['PrepUsers.with_organic_flag'].project(
            'user_login',
            'user_is_organic',
            'user_is_partner',
        )

        return streams['CollectionsRedirLog.parsed'] \
            .project(
                'fielddate', 'yandexuid', 'path', 'ui', 'page_name', 'url', 'cts', 'timestamp', 'traffic_source'
            ) \
            .groupby('yandexuid', 'fielddate') \
            .sort('timestamp', 'cts') \
            .groupby(grouping.sessions()) \
            .reduce(informer_sessions_reducer, memory_limit=8*1024) \
            .groupby('fielddate', 'user_login', 'board_slug') \
            .aggregate(
                shows_count=na.count(),
                total_useful_events=na.sum('useful_events')
            ) \
            .join(users, type='inner', by='user_login')

    @require('InformerUsefulSessions.boards')
    def stat(self, streams):
        return streams['InformerUsefulSessions.boards'] \
            .groupby('fielddate') \
            .aggregate(
                uniq_boards_shown=na.count(),
                uniq_partners_boards_shown=na.count(predicate=nf.equals('user_is_partner', True)),
                uniq_organics_boards_shown=na.count(predicate=nf.equals('user_is_organic', True)),
                uniq_boards_with_5_shows=na.count(predicate=nf.custom(lambda x: x >= 5, 'shows_count')),
                uniq_boards_with_useful_events=na.count(predicate=nf.custom(lambda x: x > 0, 'total_useful_events')),
                boards_shows_sum=na.sum('shows_count'),
                boards_useful_events_sum=na.sum('total_useful_events'),
            ) \
            .map(expand_metrics) \
            .publish(self.get_statface_report('Collections/Metrics/board2user/InformerUsefulActionsV1'), allow_change_job=True)
