# coding: utf-8

from analytics.plotter_lib.plotter import Plot, require
from nile.api.v1 import (
    with_hints,
    extended_schema,
    extractors as ne,
    aggregators as na,
    filters as nf,
    Record
)


def convert_ui(ui):
    if ui == 'yandexApp':
        return 'mobile'
    elif ui == 'tablet':
        return 'desktop'

    return ui


def extract_block_type(feed_session_id, cards_session_id, boards_session_id):
    if feed_session_id:
        return 'feed'
    elif cards_session_id:
        return 'card2card'
    elif boards_session_id:
        return 'board2board'


def extract_block_key(feed_session_id, cards_session_id, boards_session_id):
    if feed_session_id:
        return feed_session_id
    elif cards_session_id:
        return cards_session_id
    elif boards_session_id:
        return boards_session_id


def is_good_click(path):
    if path.startswith('click') and path not in ('click.kebab', 'click.card.complain'):
        return True
    else:
        return False


@with_hints(output_schema=extended_schema(clicked=int, clicks=int, shows=int))
def block_reducer(groups):
    for key, recs in groups:
        has_content = False
        outrec = {
            'fielddate': key.fielddate,
            'yandexuid': key.yandexuid,
            'ui': key.ui,
            'block_type': key.block_type,
            'block_key': key.fielddate,
            'clicked': 0,
            'clicks': 0,
            'shows': 1,
        }

        for rec in recs:
            if isinstance(rec.path, str) and is_good_click(rec.path):
                outrec['clicked'] = 1
                outrec['clicks'] += 1
            elif rec.path == 'track.objects':
                has_content = True

        if has_content:
            yield Record.from_dict(outrec)
            outrec['ui'] = '_total_'
            yield Record.from_dict(outrec)


class ItemToItemCTR(Plot):
    @require('CollectionsRedirLog.parsed')
    def item_to_item_publish(self, streams):
        return streams['CollectionsRedirLog.parsed'] \
            .project(
                'fielddate',
                'path',
                'yandexuid',
                ui=ne.custom(convert_ui, 'ui').with_type(str),
                block_type=ne.custom(extract_block_type, 'feed_session_id', 'cards_session_id', 'boards_session_id').with_type(str),
                block_key=ne.custom(extract_block_key, 'feed_session_id', 'cards_session_id', 'boards_session_id').with_type(str)
            ) \
            .filter(nf.custom(lambda x: True if x else False, 'block_key')) \
            .project(
                ne.all(('ui', 'block_type')),
                ui=ne.custom(lambda x: x if x else 'undefined', 'ui').with_type(str),
                block_type=ne.custom(lambda x: x if x else 'undefined', 'block_type').with_type(str)
            ) \
            .groupby('fielddate', 'yandexuid', 'ui', 'block_type', 'block_key') \
            .reduce(
                block_reducer
            ) \
            .groupby('fielddate', 'ui', 'block_type') \
            .aggregate(
                users=na.count_distinct('yandexuid'),
                shows=na.sum('shows'),
                clicks=na.sum('clicks'),
                clicked=na.sum('clicked')
            ) \
            .project(
                ne.all(),
                ctr=ne.custom(lambda x, y: float(x) / y, 'clicked', 'shows').with_type(float),
                clicks_by_shows=ne.custom(lambda x, y: float(x) / y, 'clicks', 'shows').with_type(float)
            ) \
            .publish(self.get_statface_report('Collections/Metrics/feed_metrics/item2item_ctrs'), allow_change_job=True)
