# coding: utf-8

import re
import urllib

from analytics.collections.plotter_collections.plots.utils import get_user_login_board_slug_from_url
from analytics.plotter_lib.plotter import Plot, require
from analytics.plotter_lib.utils import AddTotalsMapper, PrecomputeTotalsMapper

from nile.api.v1 import (
    with_hints,
    filters as nf,
    extractors as ne,
    aggregators as na,
    extended_schema
)
from qb2.api.v1 import (
    typing as qt,
    filters as qf
)


def is_url_social(url):
    social_pattern = r'.*vk\.com/|facebook\.com|ok\.ru|twitter\.com|instagram\.com'

    if isinstance(url, str):
        return bool(re.match(social_pattern, urllib.unquote(url)))
    else:
        return False


def is_phone(phone):
    phone_pattern = r'^tel:\+[0-9]+'

    if isinstance(phone, str):
        return bool(re.match(phone_pattern, urllib.unquote(phone)))
    else:
        return False


def extract_user_login(url):
    if isinstance(url, str):
        login = get_user_login_board_slug_from_url(url)[0]
        if login:
            return login
    return ''


class LeadGenerationMetric(Plot):
    @require('CollectionsRedirLog.parsed')
    def chats_with_authors(self, streams):
        """Нажатия на кнопку 'Чат с автором' на странице профиля и борды"""
        return streams['CollectionsRedirLog.parsed'] \
            .filter(
                nf.and_(
                    nf.equals('path', 'open.chat'),
                    nf.equals('chat_type', 'owner'),
                    nf.custom(lambda x: x is not None and len(x) > 0, 'owner_id')
                )
            ) \
            .project(
                'fielddate',
                'ui',
                user_id='owner_id',
                lead_type=ne.custom(lambda loc: 'open.chat.{}'.format(loc), 'loc').with_type(qt.String)
            )

    @require('CollectionsRedirLog.parsed')
    def card_polaroid_greenurls(self, streams):
        """Переходы по гринурлу с полароида карточек на странице борды
        включает в себя переходы сразу на сайт при нажатии на карточку-ссылку"""
        return streams['CollectionsRedirLog.parsed'] \
            .filter(
                nf.and_(
                    nf.equals('path', 'click.link_external'),
                    nf.equals('loc', 'board'),
                    nf.equals('page_name', 'board')
                )
            ) \
            .project(
                'fielddate',
                'ui',
                'card_id',
                lead_type=ne.const('card.from_board.link_external').with_type(qt.String)
            )

    @require('CollectionsRedirLog.parsed')
    def card_viewer_greenurl(self, streams):
        """Переходы по гринурлу из просмотрщика,
        в просмотрщик при этом можно перейти с почти любой страницы Коллекций, не только борды"""
        return streams['CollectionsRedirLog.parsed'] \
            .filter(
                nf.and_(
                    nf.equals('path', 'click.link_external'),
                    nf.equals('loc', 'viewer'),
                    nf.equals('page_name', 'card')
                )
            ) \
            .project(
                'fielddate',
                'ui',
                'card_id',
                lead_type=ne.const('card.from_viewer.link_external').with_type(qt.String)
            )

    @require(
        'LeadGenerationMetric.card_polaroid_greenurls',
        'LeadGenerationMetric.card_viewer_greenurl',
        'PrepCards.with_organic_flag'
    )
    def join_cards_with_db(self, streams):
        """Для событий лидогенерации, как переходов по гринурлам с карточек — приджоинивает user_id автора карточки"""
        return streams['LeadGenerationMetric.card_polaroid_greenurls'] \
            .concat(streams['LeadGenerationMetric.card_viewer_greenurl']) \
            .join(
                streams['PrepCards.with_organic_flag'],
                by='card_id',
                type='inner'
            ) \
            .project(
                'fielddate',
                'ui',
                'lead_type',
                'user_company_id',
                'user_type',
                user_id='card_owner_id',
            )

    @require('CollectionsRedirLog.parsed')
    def board_header_organization_greenurl(self, streams):
        """Клики по контактам лида на странице борды"""

        return streams['CollectionsRedirLog.parsed'] \
            .filter(
            nf.and_(
                nf.equals('path', 'click.link_external'),
                nf.equals('loc', 'headerOrganizationContacts'),
                nf.equals('page_name', 'board')
            )
            ) \
            .project(
                'fielddate',
                'ui',
                'board_id',
                lead_type=ne.const('board.from_header.link_external').with_type(qt.String)
            )

    @require('CollectionsRedirLog.parsed')
    def board_header_organization_phone(self, streams):
        """Клики по контактам лида на странице борды"""

        return streams['CollectionsRedirLog.parsed'] \
            .filter(
            nf.and_(
                nf.equals('path', 'click.fetch.user.phones'),
                nf.equals('loc', 'headerOrganizationContacts'),
                nf.equals('page_name', 'board')
            )
            ) \
            .project(
                'fielddate',
                'ui',
                'board_id',
                lead_type=ne.const('board.from_header.fetch.user_phone').with_type(qt.String)
            )

    @require(
        'LeadGenerationMetric.board_header_organization_greenurl',
        'LeadGenerationMetric.board_header_organization_phone',
        'PrepBoards.with_organic_flag'
    )
    def join_boards_with_db(self, streams):
        return streams['LeadGenerationMetric.board_header_organization_greenurl'] \
            .concat(streams['LeadGenerationMetric.board_header_organization_phone']) \
            .join(
                streams['PrepBoards.with_organic_flag'],
                by='board_id',
                type='inner'
            ) \
            .project(
                'fielddate',
                'ui',
                'lead_type',
                'user_company_id',
                'user_type',
                user_id='board_owner_id',
            )

    @require(
        'CollectionsRedirLog.parsed'
    )
    def def_click_call(self, streams):

        return streams['CollectionsRedirLog.parsed'] \
            .filter(
                nf.and_(
                    nf.equals('path', 'click.link'),
                    nf.or_(nf.equals('ui', 'mobile'), nf.equals('ui', 'yandexApp')),
                    nf.custom(lambda x: is_phone(x), 'href')
                )
            ) \
            .project(
                'fielddate',
                'ui',
                user_login=ne.custom(extract_user_login, 'url').with_type(str),
                lead_type=ne.const('click.call').with_type(qt.String)
            )

    @require(
        'CollectionsRedirLog.parsed'
    )
    def def_click_user_contacts_social(self, streams):

        return streams['CollectionsRedirLog.parsed'] \
            .filter(
                nf.and_(
                    nf.equals('path', 'click.link_external'),
                    nf.equals('loc', 'userContacts'),
                    nf.or_(nf.equals('ui', 'mobile'), nf.equals('ui', 'yandexApp')),
                    nf.custom(lambda x: is_url_social(x), 'href')
                )
            ) \
            .project(
                'fielddate',
                'ui',
                user_login=ne.custom(extract_user_login, 'url').with_type(str),
                lead_type=ne.const('click.user_contacts.social').with_type(qt.String)
            )

    @require(
        'CollectionsRedirLog.parsed'
    )
    def def_click_user_contacts_site(self, streams):

        return streams['CollectionsRedirLog.parsed'] \
            .filter(
                nf.and_(
                    nf.equals('path', 'click.link_external'),
                    nf.equals('loc', 'userContacts'),
                    nf.or_(nf.equals('ui', 'mobile'), nf.equals('ui', 'yandexApp')),
                    nf.custom(lambda x: not is_url_social(x), 'href')
                )
            ) \
            .project(
                'fielddate',
                'ui',
                user_login=ne.custom(extract_user_login, 'url').with_type(str),
                lead_type=ne.const('click.user_contacts.site').with_type(qt.String)
            )

    @require(
        'PrepUsers.with_organic_flag',
        'LeadGenerationMetric.def_click_call',
        'LeadGenerationMetric.def_click_user_contacts_social',
        'LeadGenerationMetric.def_click_user_contacts_site'
    )
    def join_users_with_db(self, streams):
        return self.job.concat(
            streams['LeadGenerationMetric.def_click_call'],
            streams['LeadGenerationMetric.def_click_user_contacts_social'],
            streams['LeadGenerationMetric.def_click_user_contacts_site']
            ) \
            .join(
                streams['PrepUsers.with_organic_flag'],
                by='user_login',
                type='inner'
            ) \
            .project(
                'fielddate',
                'ui',
                'lead_type',
                'user_company_id',
                'user_type',
                'user_id',
            )

    @require(
        'LeadGenerationMetric.join_cards_with_db',
        'LeadGenerationMetric.join_boards_with_db',
        'LeadGenerationMetric.join_users_with_db'
    )
    def publish(self, streams):
        joined_data = self.job.concat(
            streams['LeadGenerationMetric.join_cards_with_db'],
            streams['LeadGenerationMetric.join_boards_with_db'],
            streams['LeadGenerationMetric.join_users_with_db']
        )

        unique_users = joined_data \
            .unique('fielddate', 'ui', 'user_type', 'user_id') \
            .groupby('fielddate', 'ui', 'user_type') \
            .aggregate(
                unique_users_count=na.count()
            ) \
            .project(
                ne.all(),
                lead_type=ne.const('_total_').with_type(qt.String)
            )
        yield 'unique_users_count', unique_users

        REPORT_KEY_FIELDS = ('fielddate', 'ui', 'user_type', 'lead_type')
        leads = joined_data \
            .sort(*REPORT_KEY_FIELDS) \
            .map(
                with_hints(output_schema=extended_schema(leads_count=int))(
                    PrecomputeTotalsMapper(REPORT_KEY_FIELDS, count_key='leads_count')
                ),
                ordered=True
            ) \
            .map(with_hints(output_schema=extended_schema())(
                AddTotalsMapper(REPORT_KEY_FIELDS, ['fielddate', 'leads_count'])
            )) \
            .groupby(*REPORT_KEY_FIELDS) \
            .aggregate(
                leads_count=na.sum('leads_count')
            )
        yield 'leads_count', leads

        org_leads = joined_data \
            .filter(qf.nonzero('user_company_id')) \
            .project(ne.all('user_type'), user_type=ne.const('organization')) \
            .sort(*REPORT_KEY_FIELDS) \
            .map(
                with_hints(output_schema=extended_schema(leads_count=int))(
                    PrecomputeTotalsMapper(REPORT_KEY_FIELDS, count_key='leads_count')
                ),
                ordered=True
            ) \
            .map(with_hints(output_schema=extended_schema())(
                AddTotalsMapper(REPORT_KEY_FIELDS, ['fielddate', 'user_type', 'leads_count'])
            )) \
            .groupby(*REPORT_KEY_FIELDS) \
            .aggregate(
                leads_count=na.sum('leads_count')
            )

        yield leads \
            .concat(org_leads) \
            .join(
                unique_users,
                by=REPORT_KEY_FIELDS,
                type='left'
            ) \
            .publish(self.get_statface_report('Collections/Metrics/Partners/LeadsV1'), allow_change_job=True)
