# coding: utf-8

import re

from collections import Counter
from analytics.plotter_lib.plotter import Plot, require
from analytics.collections.plotter_collections.plots.utils import format_card_origin, mongo_id_to_datestr, DATE_FORMAT
from analytics.collections.plotter_collections.plots.utils import readable_shows_location

from nile.api.v1 import (
    Record,
    with_hints,
    extended_schema,
    aggregators as na,
    extractors as ne,
    filters as nf,
    grouping
)
from qb2.api.v1 import (
    filters as qf,
    typing as qt
)


def extract_unique_board_shows_from_sessions_reducer(groups):
    for key, records in groups:
        unique_board_shows = {}
        for r in records:
            if r.board_id and r.board_id in unique_board_shows:
                continue
            unique_board_shows[r.board_id] = r

        for board_id, show in unique_board_shows.items():
            yield show


def enrich_internal_traffic_source(groups):
    """Для внутренних переходов на карточки или борды приклеиваем location перехода"""
    for key, records in groups:
        last_click = None
        last_access = None
        accesses = {}
        for r in records:
            # Берём последний клик на карточку или борду
            if r.loc and r.path in ('click.card', 'click.board'):
                last_click = r
            # Если произошёл access на карточку/борду и последний клик был по соответствующей карточке/борде,
            # то подклеиваем location клика к источнику траффика(для внутрненних переходов)
            elif r.path == 'access' and r.board_id and r.parsed_page_name in ('card', 'board'):
                if r.traffic_source.startswith('internal') and last_click and \
                        (r.get('card_id') == last_click.get('card_id') or r.get('board_id') == last_click.get('board_id')):
                    traffic_source = '{}_{}'.format(r.traffic_source, last_click.loc)
                else:
                    traffic_source = r.traffic_source
                if last_access and last_access.board_id == r.board_id:
                    continue

                last_access = r

                if r.board_id not in accesses:
                    accesses[r.board_id] = Record(
                        board_id=r.board_id,
                        yandexuid=r.yandexuid,
                        parsed_page_name=r.parsed_page_name,
                        traffic_source=traffic_source
                    )

        for board_id, r in accesses.items():
            yield r


@with_hints(output_schema=dict(
    board_id=str,
    shows=int,
    card2card_shows=int,
    card2board_shows=int,
    board2board_shows=int,
    card2user_shows=int,
    board2user_shows=int,
    shows_location=qt.Yson
))
def shows_reducer(groups):
    for key, records in groups:
        shows = 0
        card2card_shows = 0
        card2board_shows = 0
        board2board_shows = 0
        card2user_shows = 0
        board2user_shows = 0
        shows_location = Counter()
        for r in records:
            shows += 1
            card2card_shows += bool(re.match(r'.*(card2card).*', r.location))
            card2board_shows += bool(re.match(r'.*(card2board).*', r.location))
            board2board_shows += bool(re.match(r'.*(board2board).*', r.location))
            card2user_shows += bool(re.match(r'.*(card2user).*', r.location))
            board2user_shows += bool(re.match(r'.*(board2user).*', r.location))
            shows_location[r.location] += 1

        yield Record(
            board_id=key.board_id,
            shows=shows,
            card2card_shows=card2card_shows,
            card2board_shows=card2board_shows,
            board2board_shows=board2board_shows,
            card2user_shows=card2user_shows,
            board2user_shows=board2user_shows,
            shows_location=shows_location
        )


@with_hints(
    output_schema=dict(
        user_id=str,
        user_url=str,
        user_login=str,
        user_name=qt.Optional[qt.String],
        user_type=str,
        board_access_count=int,
        board_with_access=qt.Yson,
        traffic_sources=qt.Yson,
        board_likes_count=int,
        board_with_likes=qt.Yson,
        board_card_likes_count=int,
        board_with_card_likes=qt.Yson,
        board_subs_count=int,
        board_with_new_subs=qt.Yson,
        new_cards=int,
        board_with_new_cards=qt.Yson,
        card_origins=qt.Yson,
        board_shows=int,
        board_with_shows=qt.Yson,
        board_corgie=int,
        board_features_corgie_v2=qt.Yson,
        board_features_corgie_v3=qt.Yson,
        user_subs_count=int
    )
)
def user_reducer(groups):
    for key, records in groups:
        outrec = {
            'user_id': key.user_id,
            'user_url': 'https://yandex.ru/collections/user/{}'.format(key.user_login),
            'user_login': key.user_login,
            'user_name': key.user_name,
            'user_type': key.user_type,
            'user_subs_count': 0,
            'board_access_count': 0,
            'board_with_access': Counter(),
            'traffic_sources': Counter(),
            'board_likes_count': 0,
            'board_with_likes': Counter(),
            'board_card_likes_count': 0,
            'board_with_card_likes': Counter(),
            'board_subs_count': 0,
            'board_with_new_subs': Counter(),
            'new_cards': 0,
            'board_with_new_cards': Counter(),
            'card_origins': Counter(),
            'board_shows': 0,
            'board_with_shows': Counter(),
            'board_corgie': 0,
            'board_features_corgie_v2': Counter(),
            'board_features_corgie_v3': Counter()
        }
        for r in records:
            outrec['board_access_count'] += r.get('access_count') if r.get('access_count') else 0
            outrec['board_with_access'][r.get('board_id')] += r.get('board_access_count') if r.get('board_access_count') else 0
            if r.get('traffic_sources'):
                for traffic_source, access in r.get('traffic_sources'):
                    outrec['traffic_sources'][traffic_source] += access
            outrec['board_likes_count'] += r.get('board_likes_count') if r.get('board_likes_count') else 0
            outrec['board_with_likes'][r.get('board_id')] += r.get('board_likes_count') if r.get('board_likes_count') else 0
            outrec['board_card_likes_count'] += r.get('board_card_likes_count') if r.get('board_card_likes_count') else 0
            outrec['board_with_card_likes'][r.get('board_id')] += r.get('board_card_likes_count') if r.get('board_card_likes_count') else 0
            outrec['board_subs_count'] += r.get('board_subs_count') if r.get('board_subs_count') else 0
            outrec['board_with_new_subs'][r.get('board_id')] += r.get('board_subs_count') if r.get('board_subs_count') else 0
            outrec['new_cards'] += r.get('new_cards') if r.get('new_cards') else 0
            outrec['board_with_new_cards'][r.get('board_id')] += r.get('new_cards') if r.get('new_cards') else 0
            if r.get('card_origins'):
                for origin, count in r.get('card_origins'):
                    outrec['card_origins'][origin] += count
            outrec['board_shows'] += r.get('shows') if r.get('shows') else 0
            if r.get('shows_location'):
                for show_location, shows in r.get('shows_location').items():
                    outrec['board_with_shows'][show_location] += shows
            if r.get('board_is_corgie_v1'):
                outrec['board_corgie'] += r.get('board_is_corgie_v1')
            if r.get('board_features_corgie_v2'):
                outrec['board_features_corgie_v2'][r.get('board_features_corgie_v2')] += 1
            if r.get('board_features_corgie_v3'):
                outrec['board_features_corgie_v3'][r.get('board_features_corgie_v3')] += 1
            if r.get('user_subs_count'):
                outrec['user_subs_count'] = r.user_subs_count

        yield Record.from_dict(outrec)


class OrgTops(Plot):
    """Класс считает просмотры, новые карточки, подписки, лайки, показы, для бордов,
     берёт по этим показателям топы и ложит в табличку"""
    @require('PrepCards.new_cards_with_organic_flag', layer='org_tops')
    def boards_with_new_cards(self, streams):
        return streams['PrepCards.new_cards_with_organic_flag'] \
            .filter(
                nf.and_(
                    nf.not_(nf.equals('card_is_banned', True)),
                    nf.not_(qf.nonzero('user_ban'))
                )
            ) \
            .project(board_id='card_board_id', card_origins=ne.custom(format_card_origin, 'card_origin_client_name', 'card_origin_action', 'card_origin_ui').with_type(str)) \
            .groupby('board_id') \
            .aggregate(
                new_cards=na.count(),
                card_origins=na.histogram('card_origins')
            )

    @require('CollectionsRedirLog.clean', layer='org_tops')
    def boards_with_traffic(self, streams):
        return streams['CollectionsRedirLog.clean'] \
            .filter(qf.nonzero('timestamp', 'yandexuid')) \
            .project(
                'board_id',
                'yandexuid',
                'cts',
                'parsed_page_name',
                'loc',
                'timestamp',
                'path',
                'traffic_source',
            ) \
            .groupby('yandexuid') \
            .sort('timestamp', 'cts') \
            .groupby(grouping.sessions()) \
            .reduce(with_hints(output_schema=dict(board_id=str, yandexuid=str, traffic_source=str, parsed_page_name=str))(enrich_internal_traffic_source)) \
            .groupby('board_id') \
            .aggregate(
                access_count=na.count(),
                internal_access_count=na.count(nf.custom(lambda x: x.startswith('internal'), 'traffic_source')),
                access_cards=na.count(nf.equals('parsed_page_name', 'card')),
                users_count=na.count_distinct('yandexuid'),
                traffic_sources=na.histogram('traffic_source')
            )

    @require('//home/collections-backups/db/prod/collections-fast-dump/latest/subscription', layer='org_tops')
    def boards_with_subs(self, streams):
        date_start = self.date.strftime(DATE_FORMAT)
        date_end = self.dateend.strftime(DATE_FORMAT)

        return streams['//home/collections-backups/db/prod/collections-fast-dump/latest/subscription'] \
            .filter(nf.and_(nf.equals('_cls', 'Subscription.BoardSubscription'), nf.custom(lambda x: not x.get('due_to_user'), 'document'))) \
            .project(board_id='target', sub_created_at=ne.custom(mongo_id_to_datestr, 'id').with_type(str)) \
            .filter(nf.custom(lambda dt: date_start <= dt <= date_end, 'sub_created_at')) \
            .groupby('board_id') \
            .aggregate(
                board_subs_count=na.count()
            )

    @require('//home/collections-backups/db/prod/collections-fast-dump/latest/subscription', layer='org_tops')
    def users_with_subs(self, streams):
        date_start = self.date.strftime(DATE_FORMAT)
        date_end = self.dateend.strftime(DATE_FORMAT)

        return streams['//home/collections-backups/db/prod/collections-fast-dump/latest/subscription'] \
            .filter(nf.equals('_cls', 'Subscription.UserSubscription')) \
            .project(user_id='target', sub_created_at=ne.custom(mongo_id_to_datestr, 'id').with_type(str)) \
            .filter(nf.custom(lambda dt: date_start <= dt <= date_end, 'sub_created_at')) \
            .groupby('user_id') \
            .aggregate(
                user_subs_count=na.count()
            )

    @require('//home/collections-backups/db/prod/collections-fast-dump/latest/board_like', layer='org_tops')
    def boards_with_likes(self, streams):
        date_start = self.date.strftime(DATE_FORMAT)
        date_end = self.dateend.strftime(DATE_FORMAT)

        return streams['//home/collections-backups/db/prod/collections-fast-dump/latest/board_like'] \
            .project(board_id='board', like_created_at=ne.custom(mongo_id_to_datestr, 'id').with_type(str)) \
            .filter(nf.custom(lambda dt: date_start <= dt <= date_end, 'like_created_at')) \
            .groupby('board_id') \
            .aggregate(
                board_likes_count=na.count()
            )

    @require('//home/collections-backups/db/prod/collections-fast-dump/latest/card_like', 'Cards.parsed', layer='org_tops')
    def boards_with_card_likes(self, streams):
        date_start = self.date.strftime(DATE_FORMAT)
        date_end = self.dateend.strftime(DATE_FORMAT)

        return streams['//home/collections-backups/db/prod/collections-fast-dump/latest/card_like'] \
            .project(card_id='card', like_created_at=ne.custom(mongo_id_to_datestr, 'id').with_type(str)) \
            .filter(nf.custom(lambda dt: date_start <= dt <= date_end, 'like_created_at')) \
            .join(streams['Cards.parsed'].project('card_id', board_id='card_board_id'), by='card_id', type='inner') \
            .groupby('board_id') \
            .aggregate(
                board_card_likes_count=na.count(),
                card_likes=na.histogram('card_id')
            )

    @require('AllServicesCollectionsShows.log', layer='org_tops')
    def boards_with_shows(self, streams):
        return streams['AllServicesCollectionsShows.log'] \
            .groupby('yandexuid') \
            .sort('timestamp') \
            .groupby(grouping.sessions()) \
            .reduce(with_hints(output_schema=extended_schema())(extract_unique_board_shows_from_sessions_reducer)) \
            .filter(qf.nonzero('board_id')) \
            .project(
                'board_id',
                ui=ne.custom(lambda x: x if x else 'empty', 'ui').with_type(str),
                location=ne.custom(readable_shows_location, 'service', 'page_name', 'location', 'content_type',
                                   'content_subtype').with_type(str)
            ) \
            .groupby('board_id') \
            .reduce(shows_reducer)

    @require(
        'PrepUsers.with_organic_flag',
        'Boards.parsed',
        'Corgie.corgie',
        'OrgTops.boards_with_new_cards',
        'OrgTops.boards_with_traffic',
        'OrgTops.boards_with_subs',
        'OrgTops.users_with_subs',
        'OrgTops.boards_with_likes',
        'OrgTops.boards_with_card_likes',
        'OrgTops.boards_with_shows',
        'TimeSpent.board_card_time_spent',
        layer='org_tops'
    )
    def joined_stats(self, streams):
        return streams['Boards.parsed'] \
            .filter(nf.not_(nf.equals('board_is_banned', True))) \
            .join(
                streams['PrepUsers.with_organic_flag'].filter(nf.not_(qf.nonzero('user_ban'))),
                type='inner',
                by_left='board_owner_id',
                by_right='user_id'
            ) \
            .project(
                ne.all(),
                board_url=ne.custom(lambda x, y: 'https://yandex.ru/collections/user/{}/{}/'.format(x, y), 'user_login', 'board_slug').with_type(str)
            ) \
            .join(
                streams['Corgie.corgie'].project('board_id', board_is_corgie_v1=ne.const(True)),
                type='left',
                by='board_id'
            ) \
            .join(
                streams['OrgTops.boards_with_new_cards'],
                type='left',
                by='board_id'
            ) \
            .join(
                streams['OrgTops.boards_with_traffic'],
                type='left',
                by='board_id'
            ) \
            .join(
                streams['OrgTops.boards_with_subs'],
                type='left',
                by='board_id'
            ) \
            .join(
                streams['OrgTops.users_with_subs'],
                type='left',
                by='user_id'
            ) \
            .join(
                streams['OrgTops.boards_with_likes'],
                type='left',
                by='board_id'
            ) \
            .join(
                streams['OrgTops.boards_with_card_likes'],
                type='left',
                by='board_id'
            ) \
            .join(
                streams['OrgTops.boards_with_shows'],
                type='left',
                by='board_id'
            ) \
            .join(
                streams['TimeSpent.board_card_time_spent'],
                type='left',
                by='board_id'
            )

    @require('OrgTops.joined_stats', layer='org_tops')
    def top_org_boards(self, streams):
        return self.job \
            .concat(
                streams['OrgTops.joined_stats'].filter(qf.nonzero('access_count')).top(10, by='access_count'),
                streams['OrgTops.joined_stats'].filter(qf.nonzero('board_likes_count')).top(10, by='board_likes_count'),
                streams['OrgTops.joined_stats'].filter(qf.nonzero('board_card_likes_count')).top(10, by='board_card_likes_count'),
                streams['OrgTops.joined_stats'].filter(qf.nonzero('board_subs_count')).top(10, by='board_subs_count'),
                streams['OrgTops.joined_stats'].filter(qf.nonzero('board_is_corgie_v1', 'new_cards')).top(10, by='new_cards'),
                streams['OrgTops.joined_stats'].filter(qf.nonzero('card2user_shows')).top(10, by='card2user_shows'),
                streams['OrgTops.joined_stats'].filter(qf.nonzero('board2user_shows')).top(10, by='board2user_shows'),
                streams['OrgTops.joined_stats'].filter(qf.nonzero('board2board_shows')).top(10, by='board2board_shows'),
                streams['OrgTops.joined_stats'].filter(qf.nonzero('card2board_shows')).top(10, by='card2board_shows'),
                streams['OrgTops.joined_stats'].filter(qf.nonzero('card2card_shows')).top(10, by='card2card_shows'),

                streams['OrgTops.joined_stats'].filter(nf.and_(qf.nonzero('access_count'), nf.equals('user_type', 'company'))).top(10, by='access_count'),
                streams['OrgTops.joined_stats'].filter(nf.and_(qf.nonzero('board_likes_count'), nf.equals('user_type', 'company'))).top(10, by='board_likes_count'),
                streams['OrgTops.joined_stats'].filter(nf.and_(qf.nonzero('board_card_likes_count'), nf.equals('user_type', 'company'))).top(10, by='board_card_likes_count'),
                streams['OrgTops.joined_stats'].filter(nf.and_(qf.nonzero('board_subs_count'), nf.equals('user_type', 'company'))).top(10, by='board_subs_count'),
                streams['OrgTops.joined_stats'].filter(nf.and_(qf.nonzero('board_is_corgie_v1', 'new_cards'), nf.equals('user_type', 'company'))).top(10, by='new_cards'),
                streams['OrgTops.joined_stats'].filter(nf.and_(qf.nonzero('card2user_shows'), nf.equals('user_type', 'company'))).top(10, by='card2user_shows'),
                streams['OrgTops.joined_stats'].filter(nf.and_(qf.nonzero('board2user_shows'), nf.equals('user_type', 'company'))).top(10, by='board2user_shows'),
                streams['OrgTops.joined_stats'].filter(nf.and_(qf.nonzero('board2board_shows'), nf.equals('user_type', 'company'))).top(10, by='board2board_shows'),
                streams['OrgTops.joined_stats'].filter(nf.and_(qf.nonzero('card2board_shows'), nf.equals('user_type', 'company'))).top(10, by='card2board_shows'),
                streams['OrgTops.joined_stats'].filter(nf.and_(qf.nonzero('card2card_shows'), nf.equals('user_type', 'company'))).top(10, by='card2card_shows'),
            ) \
            .put('//home/collections/analytics/backups/org_tops/top_boards')

    @require('OrgTops.joined_stats', 'OrgTops.users_with_subs', layer='org_tops')
    def prep_users_stats(self, streams):
        return streams['OrgTops.joined_stats'] \
            .filter(qf.nonzero('user_id', 'board_is_corgie_v1')) \
            .groupby('user_id', 'user_login', 'user_name', 'user_type') \
            .reduce(user_reducer)

    @require('OrgTops.prep_users_stats', layer='org_tops')
    def top_org_users(self, streams):
        return self.job \
            .concat(
                streams['OrgTops.prep_users_stats'].filter(qf.nonzero('new_cards')).top(10, by='new_cards'),
                streams['OrgTops.prep_users_stats'].filter(qf.nonzero('user_subs_count')).top(10, by='user_subs_count'),
                streams['OrgTops.prep_users_stats'].filter(nf.and_(qf.nonzero('new_cards'), nf.equals('user_type', 'company'))).top(10, by='new_cards'),
                streams['OrgTops.prep_users_stats'].filter(nf.and_(qf.nonzero('user_subs_count'), nf.equals('user_type', 'company'))).top(10, by='user_subs_count')
            ) \
            .put('//home/collections/analytics/backups/org_tops/top_users')
