# coding: utf-8

from urllib import unquote

from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from analytics.plotter_lib.utils import fromtimestamp, parse_urlencoded_json
from analytics.collections.plotter_collections.plots.utils import get_collections_page
from traffic_source import get_traffic_source

from nile.api.v1 import (
    Record,
    with_hints,
    filters as nf
)
from qb2.api.v1 import (
    typing as qt,
    resources,
)


def parse_testids(testids):
    result = []
    splitted = testids.split(';')
    for item in splitted:
        if item:
            parts = item.replace('%2C', ',').split(',')
            try:
                testid = int(parts[0])
                result.append(testid)
            except:
                pass
    return result


class ParseRedirLog(object):
    def __init__(self, redir_fields, uatraits_resource):
        self.redir_fields = redir_fields
        self.uatraits_resource = uatraits_resource

    def __call__(self, records):
        for record in records:
            if record.dict.get('env') == 'testing' or not isinstance(record.dict.get('url'), str):
                continue

            data = {
                'fielddate': fromtimestamp(float(record.dict.get('timestamp', 0))).strftime(DATE_FORMAT),
                'referer': unquote(record.dict.get('referer', '')),
                'json_info': parse_urlencoded_json(record.dict.get('jsonInfo')),
                'testids': parse_testids(record.dict.get('tids', ''))
            }
            for key, raw_key in self.redir_fields.items():
                data[key] = record.dict.get(raw_key)

            if record.dict.get('url'):
                page = get_collections_page(record.dict.get('url'))
                if page:
                    data['parsed_page_name'] = page

            try:
                data['timestamp'] = int(data['timestamp'])
            except:
                data['timestamp'] = None
            try:
                detector = resources.get(self.uatraits_resource)
                parsed_uatraits = detector.detect(record.dict.get('user_agent'))
                data['browser_name'] = parsed_uatraits.get('BrowserName')
            except:
                data['browser_name'] = None

            data['traffic_source'] = get_traffic_source(data['url'], data['referer'], data['browser_name'])
            if data['traffic_source'] == 'autotests':
                continue

            yield Record.from_dict(data)


class CollectionsRedirLog(Plot):
    REDIR_FIELDS = {
        'icookie': 'icookie',
        'yandexuid': 'yandexuid',
        'puid': 'uid',
        'url': 'url',
        'timestamp': 'timestamp',
        'cts': 'u',
        'iso_eventtime': 'iso_eventtime',
        's_cards': 'sCards',
        'card_id': 'cardId',
        'board_id': 'boardId',
        'loc': 'loc',
        'page_name': 'pageName',
        'path': 'path',
        'ui': 'ui',
        'platform': 'platform',
        'user_id': 'userId',
        'user_login': 'userLogin',
        'owner_id': 'ownerId',
        'chat_type': 'chatType',
        'reason': 'reason',
        'type': 'type',
        'key': 'key',
        'tooltip_type': 'tooltipType',
        'action_type': 'actionType',
        'user_auth': 'userAuth',
        'subscribed': 'subscribed',
        'liked': 'liked',
        'feed_session_id': 'feedSessionId',
        'cards_session_id': 'cardsSessionId',
        'boards_session_id': 'boardsSessionId',
        'card_boards_session_id': 'cardBoardsSessionId',
        'req_id': 'reqid',
        'user_agent': 'user_agent',
        'ar': 'ar',
        'is_owner': 'isOwner',
        'is_my_company': 'isMyCompany',
        'href': 'href'
    }

    @require('collections-redir-log')
    def raw(self):
        return {}

    @require('CollectionsRedirLog.raw')
    def full_with_additional_days(self, streams):
        """collections-redir-log распаршенный за период с [datestart, dateend], либо от [dateend-additional_days, dateend]"""
        stream = streams['CollectionsRedirLog.raw']

        schema = {x: qt.Optional[qt.String] for x in self.REDIR_FIELDS.keys()}
        schema['fielddate'] = qt.Optional[qt.String]
        schema['referer'] = qt.Optional[qt.String]
        schema['browser_name'] = qt.Optional[qt.String]
        schema['traffic_source'] = qt.Optional[qt.String]
        schema['testids'] = qt.Optional[qt.Yson]
        schema['json_info'] = qt.Optional[qt.Yson]
        schema['parsed_page_name'] = qt.Optional[qt.String]
        schema['timestamp'] = qt.Optional[qt.Integer]
        return stream \
            .map(
                with_hints(output_schema=schema)(
                    ParseRedirLog(self.REDIR_FIELDS, resources.resource('UATraits'))
                )
            )

    @require('CollectionsRedirLog.full_with_additional_days')
    def parsed(self, streams):
        """collections-redir-log распаршенный, за весь период с [datestart, dateend]
        Имена колонок преобразованы из camelCase и snake_case (feedSessionId -> feed_session_id)

        Дополнительно содержит поля:
          * fielddate - дата в формате YYYY-MM-DD
          * browser_name - браузер пользователя, вытащенный из user_agent с помощью ua_traits
        """
        date = self.date.strftime(DATE_FORMAT)
        dateend = self.dateend.strftime(DATE_FORMAT)

        return streams['CollectionsRedirLog.full_with_additional_days'] \
            .filter(nf.custom(lambda x: date <= x <= dateend, 'fielddate'))

    @require('CollectionsRedirLog.full_with_additional_days')
    def one_day(self, streams):
        """collections-redir-log распаршенный, за один день (dateend)"""
        dateend = self.dateend.strftime(DATE_FORMAT)
        return streams['CollectionsRedirLog.full_with_additional_days'] \
            .filter(nf.equals('fielddate', dateend))

    @require('CollectionsRedirLog.parsed', 'PrepUsers.banned')
    def clean(self, streams):
        return streams['CollectionsRedirLog.parsed'] \
            .join(streams['PrepUsers.banned'], type='left_only', by='user_id')
