# coding: utf-8

from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from analytics.plotter_lib.utils import date_range, get_dts_delta, AddTotalsMapper

from nile.api.v1 import (
    aggregators as na,
    extractors as ne,
    filters as nf,
    with_hints,
    Record,
    extended_schema,
    grouping
)
from qb2.api.v1 import (
    filters as qf,
    typing as qt,
)


@with_hints(output_schema=dict(fielddate=str, board_id=str, page_name=str, ui=str))
def sessions_shows(groups):
    for key, records in groups:
        boards_views = set()
        for r in records:
            if r.path == 'access' and r.board_id:
                boards_views.add((r.board_id, r.page_name))

        for board_view in boards_views:
            yield Record(fielddate=key.fielddate, board_id=board_view[0], page_name=board_view[1], ui=key.ui)


class PrepFreshBoardsReducer(object):
    def __init__(self, date_start, date_end):
        self.date_start = date_start
        self.date_end = date_end

    def __call__(self, groups):
        for key, records in groups:
            for r in records:
                # записи для расчёта просмотров от даты добавления карточки в коллекцию
                if self.date_start <= r.card_created_at_date <= self.date_end:
                    outrec = Record(
                        board_id=r.board_id,
                        board_timestamp=r.card_created_at_timestamp,
                        fielddate=r.card_created_at_date,
                        user_type=r.user_type,
                        board_type='updated_board'
                    )
                    yield outrec
                    yield outrec.transform(user_type='_total_')
                    if r.user_company_id:
                        yield outrec.transform(user_type='company')
                    if r.board_features_corgie_v2:
                        yield outrec.transform(board_type='updated_corgie_v2')
                        yield outrec.transform(board_type='updated_corgie_v2', user_type='_total_')
                    if r.board_features_corgie_v3:
                        yield outrec.transform(board_type='updated_corgie_v3')
                        yield outrec.transform(board_type='updated_corgie_v3', user_type='_total_')

            # записи для подсчёта просмотров от даты создания коллекции
            if self.date_start <= r.board_created_at_date <= self.date_end:
                outrec = Record(
                    board_id=r.board_id,
                    board_timestamp=r.board_created_at_timestamp,
                    fielddate=r.board_created_at_date,
                    user_type=r.user_type,
                    board_type='new_board'
                )
                yield outrec
                yield outrec.transform(user_type='_total_')
                if r.user_company_id:
                    yield outrec.transform(user_type='company')
                if r.board_features_corgie_v2:
                    yield outrec.transform(board_type='new_corgie_v2')
                    yield outrec.transform(board_type='new_corgie_v2', user_type='_total_')
                if r.board_features_corgie_v3:
                    yield outrec.transform(board_type='new_corgie_v3')
                    yield outrec.transform(board_type='new_corgie_v3', user_type='_total_')


class AddPartnersZeroShows(object):
    def __init__(self, date, dateend):
        self.date = date
        self.dateend = dateend if dateend else self.date

    def __call__(self, records):
        for r in records:
            for date in date_range(self.date, self.dateend):
                date_str = date.strftime(DATE_FORMAT)
                if date_str >= r.user_created_date:
                    yield Record(
                        fielddate=date_str,
                        content_owner_id=r.user_id,
                        shows_count=0,
                    )


class ActiveUsers(object):
    def __init__(self, date, dateend):
        self.date = date
        self.dateend = dateend if dateend else self.date

    def __call__(self, groups):
        for key, records in groups:
            act_dates = set()
            for r in records:
                for date in date_range(self.date, self.dateend):
                    date_str = date.strftime(DATE_FORMAT)
                    if abs(get_dts_delta(date_str, r.fielddate).days) <= 7:
                        act_dates.add(date_str)

            for date in act_dates:
                yield Record(
                    fielddate=date,
                    rubric=key.rubric,
                    active_users_count=1
                )


def extrac_user_rubrics_names(records):
    for r in records:
        if isinstance(r.get('user_rubrics_names'), list):
            for rubric in r.get('user_rubrics_names'):
                if rubric:
                    yield r.transform(rubric=rubric)
        yield r.transform(rubric='_total_')


class Partners(Plot):
    @require('Users.partners', 'Cards.parsed')
    def get_cards(self, streams):
        return streams['Users.partners'] \
            .join(
                streams['Cards.parsed'].project('card_id', 'card_owner_id', 'card_created_at_date'),
                type='inner',
                by_left='user_id',
                by_right='card_owner_id',
            )

    @require('Users.partners', 'Boards.parsed')
    def get_boards(self, streams):
        boards = streams['Boards.parsed'] \
            .filter(
                nf.equals('board_is_default', False),
            ) \
            .project('board_id', 'board_owner_id', 'board_created_at_date')

        return streams['Users.partners'] \
            .join(
                boards,
                type='inner',
                by_left='user_id',
                by_right='board_owner_id',
            )

    @require('Users.partners')
    def users_count(self, streams):
        return streams['Users.partners'] \
            .project('user_created_date', 'user_rubrics_names') \
            .map(with_hints(output_schema=extended_schema(rubric=str))(extrac_user_rubrics_names)) \
            .groupby('user_created_date', 'rubric') \
            .aggregate(
                users_count=na.count()
            ) \
            .project(
                'users_count',
                'rubric',
                fielddate='user_created_date'
            )

    @require('Partners.get_cards')
    def cards_count(self, streams):
        return streams['Partners.get_cards'] \
            .project('card_created_at_date', 'user_rubrics_names') \
            .map(with_hints(output_schema=extended_schema(rubric=str))(extrac_user_rubrics_names)) \
            .groupby('card_created_at_date', 'rubric') \
            .aggregate(
                cards_count=na.count()
            ) \
            .project(
                'cards_count',
                'rubric',
                fielddate='card_created_at_date'
            )

    @require('Partners.get_boards')
    def boards_count(self, streams):
        return streams['Partners.get_boards'] \
            .project('board_created_at_date', 'user_rubrics_names') \
            .map(with_hints(output_schema=extended_schema(rubric=str))(extrac_user_rubrics_names)) \
            .groupby('board_created_at_date', 'rubric') \
            .aggregate(
                boards_count=na.count()
            ) \
            .project(
                'boards_count',
                'rubric',
                fielddate='board_created_at_date'
            )

    @require('Partners.get_cards', 'Partners.get_boards')
    def active_users_count(self, streams):
        cards = streams['Partners.get_cards'] \
            .project('user_rubrics_names', user_id='card_owner_id', fielddate='card_created_at_date') \
            .map(with_hints(output_schema=extended_schema(rubric=str))(extrac_user_rubrics_names))

        boards = streams['Partners.get_boards'] \
            .project('user_rubrics_names', user_id='board_owner_id', fielddate='board_created_at_date') \
            .map(with_hints(output_schema=extended_schema(rubric=str))(extrac_user_rubrics_names))

        return cards.concat(boards) \
            .groupby('user_id', 'rubric') \
            .reduce(
                with_hints(output_schema=dict(
                    fielddate=qt.String,
                    rubric=qt.String,
                    active_users_count=qt.UInt64
                ))(
                    ActiveUsers(self.date, self.dateend)
                )
            )

    @require('Users.partners', 'Corgie.corgie')
    def corgie_count(self, streams):
        return streams['Users.partners'] \
            .join(
                streams['Corgie.corgie'].project('board_id', 'board_owner_id', 'good_board_fielddate'),
                type='inner',
                by_left='user_id',
                by_right='board_owner_id',
            ) \
            .project('user_rubrics_names', 'good_board_fielddate') \
            .map(with_hints(output_schema=extended_schema(rubric=str))(extrac_user_rubrics_names)) \
            .groupby('good_board_fielddate', 'rubric') \
            .aggregate(
                corgie_count=na.count()
            ) \
            .project(
                'corgie_count',
                'rubric',
                fielddate='good_board_fielddate'
            )

    @require(
        'Partners.users_count',
        'Partners.cards_count',
        'Partners.boards_count',
        'Partners.corgie_count',
        'Partners.active_users_count'
    )
    def stat(self, streams):
        date_start = self.date.strftime(DATE_FORMAT)
        date_end = self.dateend.strftime(DATE_FORMAT)

        return self.job.concat(*streams.values()) \
            .filter(nf.custom(lambda dt: date_start <= dt <= date_end, 'fielddate')) \
            .groupby('fielddate', 'rubric') \
            .aggregate(
                users_count=na.sum('users_count'),
                cards_count=na.sum('cards_count'),
                boards_count=na.sum('boards_count'),
                corgie_count=na.sum('corgie_count'),
                active_users_count=na.sum('active_users_count')
            ) \
            .project(
                'fielddate',
                'rubric',
                users_count=ne.custom(lambda x: x if x else 0, 'users_count').with_type(int),
                cards_count=ne.custom(lambda x: x if x else 0, 'cards_count').with_type(int),
                boards_count=ne.custom(lambda x: x if x else 0, 'boards_count').with_type(int),
                corgie_count=ne.custom(lambda x: x if x else 0, 'corgie_count').with_type(int),
                active_users_count=ne.custom(lambda x: x if x else 0, 'active_users_count').with_type(int)
            ) \
            .publish(self.get_statface_report('Collections/Metrics/Partners/PartnersStatV3'), allow_change_job=True)

    @require(
        'AllServicesCollectionsShows.joined_with_db',
        'Users.partners',
        layer='sessions',
    )
    def partners_shows(self, streams):
        partners_shows = streams['AllServicesCollectionsShows.joined_with_db'] \
            .filter(nf.equals('is_owner_partner', True)) \
            .groupby('fielddate', 'content_owner_id') \
            .aggregate(
                shows_count=na.count()
            )

        partners_users = streams['Users.partners'] \
            .map(
                with_hints(output_schema=dict(
                    fielddate=qt.String,
                    content_owner_id=qt.String,
                    shows_count=qt.UInt64,
                ))(
                    AddPartnersZeroShows(self.date, self.dateend)
                )
            )

        return partners_shows \
            .concat(partners_users) \
            .groupby('fielddate', 'content_owner_id') \
            .aggregate(
                shows_count=na.sum('shows_count'),
            )

    @require(
        'Partners.partners_shows',
        layer='sessions',
    )
    def partners_shows_publish_percentile(self, streams):
        partners_count_notzero = streams['Partners.partners_shows'] \
            .filter(qf.nonzero('shows_count')) \
            .groupby('fielddate') \
            .aggregate(
                partners_shown_count=na.count(),
                nonzero_percentiles=na.quantile('shows_count', [0.05, 0.1, 0.2, 0.25, 0.5, 0.75, 0.8, 0.9, 0.95]),
            )

        return streams['Partners.partners_shows'] \
            .groupby('fielddate') \
            .aggregate(
                total_shows_count=na.sum('shows_count'),
                max_partners_shows_count=na.max('shows_count'),
                avg_shows_count=na.mean('shows_count'),
                percentiles=na.quantile('shows_count', [0.05, 0.1, 0.2, 0.25, 0.5, 0.75, 0.8, 0.9, 0.95]),
                partners_count=na.count(),
            ) \
            .join(partners_count_notzero, type='left', by='fielddate') \
            .project(
                'fielddate', 'total_shows_count', 'max_partners_shows_count', 'avg_shows_count',
                'partners_count', 'partners_shown_count',

                p5=ne.custom(lambda array:  array[0][1], 'percentiles').with_type(qt.Float),
                p10=ne.custom(lambda array: array[1][1], 'percentiles').with_type(qt.Float),
                p20=ne.custom(lambda array: array[2][1], 'percentiles').with_type(qt.Float),
                p25=ne.custom(lambda array: array[3][1], 'percentiles').with_type(qt.Float),
                p50=ne.custom(lambda array: array[4][1], 'percentiles').with_type(qt.Float),
                p75=ne.custom(lambda array: array[5][1], 'percentiles').with_type(qt.Float),
                p80=ne.custom(lambda array: array[6][1], 'percentiles').with_type(qt.Float),
                p90=ne.custom(lambda array: array[7][1], 'percentiles').with_type(qt.Float),
                p95=ne.custom(lambda array: array[8][1], 'percentiles').with_type(qt.Float),

                nonzero_p5=ne.custom(lambda array:  array[0][1], 'nonzero_percentiles').with_type(qt.Float),
                nonzero_p10=ne.custom(lambda array: array[1][1], 'nonzero_percentiles').with_type(qt.Float),
                nonzero_p20=ne.custom(lambda array: array[2][1], 'nonzero_percentiles').with_type(qt.Float),
                nonzero_p25=ne.custom(lambda array: array[3][1], 'nonzero_percentiles').with_type(qt.Float),
                nonzero_p50=ne.custom(lambda array: array[4][1], 'nonzero_percentiles').with_type(qt.Float),
                nonzero_p75=ne.custom(lambda array: array[5][1], 'nonzero_percentiles').with_type(qt.Float),
                nonzero_p80=ne.custom(lambda array: array[6][1], 'nonzero_percentiles').with_type(qt.Float),
                nonzero_p90=ne.custom(lambda array: array[7][1], 'nonzero_percentiles').with_type(qt.Float),
                nonzero_p95=ne.custom(lambda array: array[8][1], 'nonzero_percentiles').with_type(qt.Float),
            ) \
            .publish(self.get_statface_report('Collections/Metrics/Partners/PartnersShowsPercentiles'), allow_change_job=True)

    @require('PrepBoards.with_organic_flag', 'CollectionsRedirLog.parsed')
    def org_boards_shows(self, streams):
        org_boards = streams['PrepBoards.with_organic_flag'] \
            .filter(
                nf.not_(nf.equals('board_is_default', True)),
                qf.nonzero('user_company_id')) \
            .project('board_id')

        return streams['CollectionsRedirLog.parsed'] \
            .groupby('yandexuid', 'fielddate', 'ui') \
            .sort('timestamp', 'cts') \
            .groupby(grouping.sessions()) \
            .reduce(sessions_shows) \
            .join(org_boards, by='board_id', type='inner') \
            .map(with_hints(output_schema=extended_schema())(AddTotalsMapper(['fielddate', 'ui', 'page_name'], ['fielddate']))) \
            .groupby('fielddate', 'ui', 'page_name') \
            .aggregate(
                views=na.count()
            ) \
            .project(ne.all(), page_type=ne.const('org_boards')) \
            .publish(self.get_statface_report('Collections/Metrics/Partners/PageViewsV1'))

    @staticmethod
    def calc_boards_shows(redir_shows, boards, KEY_FIELDS):
        return redir_shows \
            .join(boards, by='board_id', type='right') \
            .project(
                ne.all(['show_date', 'timestamp']),
                show_date=ne.custom(lambda x: x if x else '', 'show_date').with_type(str),
                timestamp=ne.custom(lambda x: x if x else 0, 'timestamp').with_type(int),
            ) \
            .groupby(*(KEY_FIELDS + ['board_id'])) \
            .aggregate(
                first_day_shows=na.count(
                    nf.and_(
                        nf.custom(lambda x, y: x == y, 'fielddate', 'show_date'),
                        nf.custom(lambda x, y: 0 <= y - x <= 60 * 60 * 24, 'board_timestamp', 'timestamp')
                    )
                ),
                first_24hours_shows=na.count(nf.custom(lambda x, y: 0 <= y - x <= 60 * 60 * 24, 'board_timestamp', 'timestamp')),
                first_48hours_shows=na.count(nf.custom(lambda x, y: 0 <= y - x <= 60 * 60 * 24 * 2, 'board_timestamp', 'timestamp')),
                first_week_shows=na.count(nf.custom(lambda x, y: 0 <= y - x <= 60 * 60 * 24 * 7, 'board_timestamp', 'timestamp')),
                first_2weeks_shows=na.count(nf.custom(lambda x, y: 0 <= y - x <= 60 * 60 * 24 * 14, 'board_timestamp', 'timestamp')),
                first_1month_shows=na.count(nf.custom(lambda x, y: 0 <= y - x <= 60 * 60 * 24 * 30, 'board_timestamp', 'timestamp')),
            ) \
            .project(
                ne.all(['first_day_shows', 'first_24hours_shows', 'first_48hours_shows', 'first_week_shows', 'first_2weeks_shows', 'first_1month_shows']),
                first_day_shows=ne.custom(lambda x: x if x else 0, 'first_day_shows').with_type(int),
                first_24hours_shows=ne.custom(lambda x: x if x else 0, 'first_24hours_shows').with_type(int),
                first_48hours_shows=ne.custom(lambda x: x if x else 0, 'first_48hours_shows').with_type(int),
                first_week_shows=ne.custom(lambda x: x if x else 0, 'first_week_shows').with_type(int),
                first_2weeks_shows=ne.custom(lambda x: x if x else 0, 'first_2weeks_shows').with_type(int),
                first_1month_shows=ne.custom(lambda x: x if x else 0, 'first_1month_shows').with_type(int),
            ) \
            .groupby(*KEY_FIELDS) \
            .aggregate(
                boards=na.count(),
                first_day_shows=na.sum('first_day_shows'),
                first_24hours_shows=na.sum('first_24hours_shows'),
                first_48hours_shows=na.sum('first_48hours_shows'),
                first_week_shows=na.sum('first_week_shows'),
                first_2weeks_shows=na.sum('first_2weeks_shows'),
                first_1month_shows=na.sum('first_1month_shows'),

                first_day_shows_mean=na.mean('first_day_shows'),
                first_24hours_shows_mean=na.mean('first_24hours_shows'),
                first_48hours_shows_mean=na.mean('first_48hours_shows'),
                first_week_shows_mean=na.mean('first_week_shows'),
                first_2weeks_shows_mean=na.mean('first_2weeks_shows'),
                first_1month_shows_mean=na.mean('first_1month_shows'),

                first_day_shows_quantile=na.quantile('first_day_shows',  [0.05, 0.1, 0.2, 0.25, 0.5, 0.75, 0.8, 0.9, 0.95, 1]),
                first_24hours_shows_quantile=na.quantile('first_24hours_shows', [0.05, 0.1, 0.2, 0.25, 0.5, 0.75, 0.8, 0.9, 0.95, 1]),
                first_48hours_shows_quantile=na.quantile('first_48hours_shows', [0.5, 0.75, 0.95]),
                first_week_shows_quantile=na.quantile('first_week_shows', [0.05, 0.1, 0.2, 0.25, 0.5, 0.75, 0.8, 0.9, 0.95, 1]),
                first_2weeks_shows_quantile=na.quantile('first_2weeks_shows', [0.5, 0.75, 0.95]),
                first_1month_shows_quantile=na.quantile('first_1month_shows', [0.5, 0.75, 0.95]),
            ) \
            .project(
                'boards',

                'first_day_shows',
                'first_24hours_shows',
                'first_48hours_shows',
                'first_week_shows',
                'first_2weeks_shows',
                'first_1month_shows',

                'first_day_shows_mean',
                'first_24hours_shows_mean',
                'first_48hours_shows_mean',
                'first_week_shows_mean',
                'first_2weeks_shows_mean',
                'first_1month_shows_mean',

                *KEY_FIELDS,

                first_day_shows_p5=ne.custom(lambda array: array[0][1], 'first_day_shows_quantile').with_type(qt.Float),
                first_day_shows_10=ne.custom(lambda array: array[1][1], 'first_day_shows_quantile').with_type(qt.Float),
                first_day_shows_20=ne.custom(lambda array: array[2][1], 'first_day_shows_quantile').with_type(qt.Float),
                first_day_shows_p25=ne.custom(lambda array: array[3][1], 'first_day_shows_quantile').with_type(qt.Float),
                first_day_shows_p50=ne.custom(lambda array: array[4][1], 'first_day_shows_quantile').with_type(qt.Float),
                first_day_shows_p75=ne.custom(lambda array: array[5][1], 'first_day_shows_quantile').with_type(qt.Float),
                first_day_shows_p80=ne.custom(lambda array: array[6][1], 'first_day_shows_quantile').with_type(qt.Float),
                first_day_shows_p90=ne.custom(lambda array: array[7][1], 'first_day_shows_quantile').with_type(qt.Float),
                first_day_shows_p95=ne.custom(lambda array: array[8][1], 'first_day_shows_quantile').with_type(qt.Float),
                first_day_shows_p100=ne.custom(lambda array: array[9][1], 'first_day_shows_quantile').with_type(qt.Float),

                first_24hours_shows_p5=ne.custom(lambda array: array[0][1], 'first_24hours_shows_quantile').with_type(qt.Float),
                first_24hours_shows_10=ne.custom(lambda array: array[1][1], 'first_24hours_shows_quantile').with_type(qt.Float),
                first_24hours_shows_20=ne.custom(lambda array: array[2][1], 'first_24hours_shows_quantile').with_type(qt.Float),
                first_24hours_shows_p25=ne.custom(lambda array: array[3][1], 'first_24hours_shows_quantile').with_type(qt.Float),
                first_24hours_shows_p50=ne.custom(lambda array: array[4][1], 'first_24hours_shows_quantile').with_type(qt.Float),
                first_24hours_shows_p75=ne.custom(lambda array: array[5][1], 'first_24hours_shows_quantile').with_type(qt.Float),
                first_24hours_shows_p80=ne.custom(lambda array: array[6][1], 'first_24hours_shows_quantile').with_type(qt.Float),
                first_24hours_shows_p90=ne.custom(lambda array: array[7][1], 'first_24hours_shows_quantile').with_type(qt.Float),
                first_24hours_shows_p95=ne.custom(lambda array: array[8][1], 'first_24hours_shows_quantile').with_type(qt.Float),
                first_24hours_shows_p100=ne.custom(lambda array: array[9][1], 'first_24hours_shows_quantile').with_type(qt.Float),

                first_48hours_shows_p50=ne.custom(lambda array: array[0][1], 'first_48hours_shows_quantile').with_type(qt.Float),
                first_48hours_shows_p75=ne.custom(lambda array: array[1][1], 'first_48hours_shows_quantile').with_type(qt.Float),
                first_48hours_shows_p95=ne.custom(lambda array: array[2][1], 'first_48hours_shows_quantile').with_type(qt.Float),

                first_week_shows_p5=ne.custom(lambda array: array[0][1], 'first_week_shows_quantile').with_type(qt.Float),
                first_week_shows_10=ne.custom(lambda array: array[1][1], 'first_week_shows_quantile').with_type(qt.Float),
                first_week_shows_20=ne.custom(lambda array: array[2][1], 'first_week_shows_quantile').with_type(qt.Float),
                first_week_shows_p25=ne.custom(lambda array: array[3][1], 'first_week_shows_quantile').with_type(qt.Float),
                first_week_shows_p50=ne.custom(lambda array: array[4][1], 'first_week_shows_quantile').with_type(qt.Float),
                first_week_shows_p75=ne.custom(lambda array: array[5][1], 'first_week_shows_quantile').with_type(qt.Float),
                first_week_shows_p80=ne.custom(lambda array: array[6][1], 'first_week_shows_quantile').with_type(qt.Float),
                first_week_shows_p90=ne.custom(lambda array: array[7][1], 'first_week_shows_quantile').with_type(qt.Float),
                first_week_shows_p95=ne.custom(lambda array: array[8][1], 'first_week_shows_quantile').with_type(qt.Float),
                first_week_shows_p100=ne.custom(lambda array: array[9][1], 'first_week_shows_quantile').with_type(qt.Float),

                first_2weeks_shows_p50=ne.custom(lambda array: array[0][1], 'first_2weeks_shows_quantile').with_type(qt.Float),
                first_2weeks_shows_p75=ne.custom(lambda array: array[1][1], 'first_2weeks_shows_quantile').with_type(qt.Float),
                first_2weeks_shows_p95=ne.custom(lambda array: array[2][1], 'first_2weeks_shows_quantile').with_type(qt.Float),

                first_1month_shows_p50=ne.custom(lambda array: array[0][1], 'first_1month_shows_quantile').with_type(qt.Float),
                first_1month_shows_p75=ne.custom(lambda array: array[1][1], 'first_1month_shows_quantile').with_type(qt.Float),
                first_1month_shows_p95=ne.custom(lambda array: array[2][1], 'first_1month_shows_quantile').with_type(qt.Float)
            )

    @require(
        'Cards.parsed',
        'PrepBoards.with_organic_flag',
        'CollectionsRedirLog.full_with_additional_days'
    )
    def fresh_boards_shows(self, streams):
        date_start = self.date.strftime(DATE_FORMAT)
        date_end = self.dateend.strftime(DATE_FORMAT)

        new_card_date = streams['Cards.parsed'] \
            .groupby('card_created_at_date', 'card_board_id') \
            .aggregate(
                card_created_at_timestamp=na.min('card_created_at_timestamp')
            )

        boards = streams['PrepBoards.with_organic_flag'] \
            .join(
                new_card_date,
                by_left='board_id',
                by_right='card_board_id',
                type='inner'
            ) \
            .filter(
                nf.and_(
                    nf.not_(nf.equals('board_is_banned', True)),
                    nf.not_(nf.equals('board_is_private', True)),
                    nf.not_(nf.equals('board_is_default', True))
                )
            ) \
            .groupby('board_id') \
            .reduce(
                with_hints(output_schema=dict(board_id=str, board_timestamp=int, fielddate=str, user_type=str, board_type=str))(
                    PrepFreshBoardsReducer(date_start, date_end)
                )
            )

        redir_shows = streams['CollectionsRedirLog.full_with_additional_days'] \
            .filter(nf.and_(
                nf.equals('path', 'access'),
                nf.equals('page_name', 'board'),
                qf.nonzero('board_id'),
                nf.not_(nf.equals('is_owner', 'true')),
                nf.not_(nf.equals('is_my_company', 'true')),
                nf.not_(nf.equals('traffic_source', 'toloka')),
            )) \
            .project('board_id', 'timestamp', 'traffic_source', show_date='fielddate')

        yield 'fresh_boards_shows_v1', self.calc_boards_shows(
            redir_shows, boards, ['fielddate', 'user_type', 'board_type']
        ) \
            .publish(self.get_statface_report('Collections/Metrics/Partners/FreshBoardsShows'), allow_change_job=True)

        yield 'fresh_boards_shows_v2', self.calc_boards_shows(
            redir_shows.map(
                with_hints(output_schema=extended_schema())(
                    AddTotalsMapper(
                        ['traffic_source'],
                        ['show_date', 'board_id', 'timestamp']
                    )
                )
            ),
            boards,
            ['fielddate', 'user_type', 'traffic_source', 'board_type']
        ) \
            .publish(self.get_statface_report('Collections/Metrics/Partners/FreshBoardsShowsV2'), allow_change_job=True)
