# coding: utf-8

from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from analytics.plotter_lib.utils import date_range, split_by_dates, AddTotalsMapper
from pdb_analytics.corgie import get_slug_verdict

from nile.api.v1 import (
    Record,
    with_hints,
    extended_schema,
    filters as nf,
    extractors as ne,
    aggregators as na,
)

from qb2.api.v1 import (
    filters as qf,
    typing as qt,
    extractors as qe
)


def corgie_cards_reducer(groups):
    for key, recs in groups:
        corgie_card_count = 0
        for rec in recs:
            corgie_card_count += 1

            if corgie_card_count == 3:
                yield rec
                break


@with_hints(output_schema=extended_schema())
def expand_slugs(records):
    for r in records:
        for slug in r.board_slugs:
            if r.board_slug == slug:
                yield r
            else:
                yield Record(r, board_slug=slug)


class AccumReducer(object):
    def __init__(self, date, dateend):
        self.date = date
        self.dateend = dateend

    def __call__(self, groups):
        for key, records in groups:
            last_fielddate = ''
            boards = 0
            date_boards = {}
            for r in records:
                if r.get('boards'):
                    boards += r.boards
                if last_fielddate != r.get('fielddate'):
                    last_fielddate = r.fielddate
                    date_boards[r.fielddate] = boards

            boards = None
            for date in date_range(self.date, self.dateend):
                if date.strftime(DATE_FORMAT) in date_boards:
                    boards = date_boards[date.strftime(DATE_FORMAT)]
                if boards is not None:
                    yield Record(
                        boards=boards,
                        user_type=key.user_type,
                        corgie_v1=key.corgie_v1,
                        corgie_v2=key.corgie_v2,
                        corgie_v3=key.corgie_v3,
                        fielddate=date.strftime(DATE_FORMAT)
                    )


def tell_if_board_is_collab_by_permissions(permissions):
    if isinstance(permissions, list) and len(permissions) > 0:
        return True
    return False


class PrepBoards(Plot):
    @require('Boards.parsed')
    def from_toloka(self, streams):
        return streams['Boards.parsed'] \
            .filter(nf.not_(nf.equals('board_moder_status', 0)))

    @require('PrepUsers.with_organic_flag', 'Boards.parsed')
    def with_organic_flag(self, streams):
        users = streams['PrepUsers.with_organic_flag'] \
            .project(
                'user_id',
                'user_type',
                'user_is_partner',
                'user_login',
                'user_created_date',
                'user_company_id',
                'user_ban',
                content_owner_puid='user_puid',
                owner_is_organic='user_is_organic',
            )

        return streams['Boards.parsed'] \
            .join(
                users,
                type='inner',
                by_left='board_owner_id',
                by_right='user_id'
            ) \
            .project(
                ne.all(['owner_is_organic', 'user_is_partner']),
                board_owner_is_organic=ne.custom(lambda x: True if x else False, 'owner_is_organic').with_type(bool),
                board_owner_is_partner=ne.custom(lambda x: True if x else False, 'user_is_partner').with_type(bool),
                board_url=ne.custom(lambda x, y: 'https://yandex.ru/collections/user/{}/{}/'.format(x, y), 'user_login', 'board_slug').with_type(str),
            )

    @require('PrepBoards.with_organic_flag')
    def expanded_slugs(self, streams):
        return streams['PrepBoards.with_organic_flag'] \
            .map(expand_slugs)

    @require('PrepBoards.with_organic_flag')
    def organic_boards(self, streams):
        boards = streams['PrepBoards.with_organic_flag']

        return boards.filter(nf.and_(nf.equals('board_owner_is_organic', True), nf.not_(qf.nonzero('user_ban'))))

    @require('PrepBoards.organic_boards', 'TagTable.parsed')
    def organic_boards_without_spam(self, streams):
        organic_boards = streams['PrepBoards.organic_boards']
        tag_table = streams['TagTable.parsed']

        public_boards, private_boards = organic_boards.filter(nf.not_(nf.equals('board_is_banned', True))) \
            .split(nf.equals('board_is_private', True))

        public_boards_without_spam = public_boards.join(
            tag_table.filter(nf.custom(lambda prob: prob > 0.9, 'board_adv_prob')),
            by='board_id',
            type='left_only',
            allow_undefined_keys=False
        )

        return public_boards_without_spam.concat(private_boards)

    @require('PrepBoards.organic_boards')
    def public_organic_boards(self, streams):
        good_boards = streams['PrepBoards.organic_boards']

        return good_boards \
            .filter(
                nf.and_(
                    nf.equals('board_is_private', False),
                    nf.equals('board_is_default', False),
                )
            )

    @require('Boards.parsed')
    def collabs(self, streams):
        boards = streams['Boards.parsed']

        return boards.filter(nf.custom(tell_if_board_is_collab_by_permissions, 'board_permissions')) \
            .project(ne.all(), board_coauthors_count=ne.custom(lambda x: len(x), 'board_permissions'))

    @require('Boards.parsed')
    def collabs_and_potential(self, streams):
        boards = streams['Boards.parsed']
        return boards.filter(
            nf.or_(
                nf.custom(tell_if_board_is_collab_by_permissions, 'board_permissions'),
                nf.equals('board_auto_approve_invite_requests', True)
            )
        ) \
        .project(
            ne.all(exclude='board_auto_approve_invite_requests'),
            qe.yql_custom('board_auto_approve_invite_requests', 'COALESCE($p0, false)', 'board_auto_approve_invite_requests')
        )

    @require('Boards.parsed')
    def potential_open_collabs(self, streams):
        return streams['Boards.parsed'] \
            .filter(nf.equals('board_auto_approve_invite_requests', True))

    @require('PrepBoards.collabs')
    def open_collabs(self, streams):
        return streams['PrepBoards.collabs'] \
            .filter(nf.equals('board_auto_approve_invite_requests', True))

    @require('PrepBoards.with_organic_flag')
    def promo_boards(self, streams):
        return streams['PrepBoards.with_organic_flag'] \
            .filter(nf.equals('board_is_promo', True)) \
            .project(
                'board_id',
                'board_url',
                ts_created='board_created_at_timestamp',
                date_created='board_created_at_date',
                passport_login='user_login',
                puid='content_owner_puid',
            ) \
            .sort('passport_login') \
            .put('//home/collections/analytics/backups/promo_boards/promo_boards_full')


class Corgie(Plot):
    @require(
        'PrepCards.organic_cards',
        'PrepBoards.public_organic_boards',
        'CollectionsRedirLog.full_with_additional_days',
    )
    def corgie(self, streams):
        """Возвращает коллекции, которые считаются Корги V1, — качественные органические коллекции
        корги содержат 3+ карточек, не дефолтное и не спамное название.
        Возвращает все корги за всё время по базе.
        Для свежих корги более точно указана дата становления коржом, как дата распривачивания, или же дата добавления 3й карточки"""

        spamers = streams['PrepBoards.public_organic_boards'] \
            .project(ne.all(), slug_verdict=ne.custom(get_slug_verdict, 'board_slug').with_type(str)) \
            .groupby('board_owner_id') \
            .aggregate(spam_count=na.count(nf.equals('slug_verdict', 'auto_spam'))) \
            .filter(nf.custom(lambda x: x >= 2, 'spam_count'))

        # для свежих коржей считаем дату good_board_fielddate, как дату unlock'а (распривачивания)
        board_unlock_dates = streams['CollectionsRedirLog.full_with_additional_days'] \
            .filter(nf.equals('path', 'finish.board.unlock')) \
            .groupby('board_id') \
            .aggregate(
                unlock_date=na.max('fielddate')
            )

        corgie_cards = streams['PrepCards.organic_cards'] \
            .filter(nf.and_(nf.not_(nf.equals('card_is_banned', True)), nf.not_(nf.equals('card_is_private', True)))) \
            .groupby('card_board_id') \
            .sort('card_created_at_datetime') \
            .reduce(
                with_hints(output_schema=extended_schema())(
                    corgie_cards_reducer
                )
            ) \
            .project(
                'card_board_id',
                card_created_at_date_for_3th_card='card_created_at_date',
            )

        return streams['PrepBoards.public_organic_boards'] \
            .project(ne.all(), slug_verdict=ne.custom(get_slug_verdict, 'board_slug').with_type(str)) \
            .filter(nf.equals('slug_verdict', 'good')) \
            .join(spamers, type='left_only', by='board_owner_id') \
            .join(
                corgie_cards,
                type='inner',
                by_left='board_id',
                by_right='card_board_id',
                allow_undefined_keys=False
            ) \
            .join(board_unlock_dates, type='left', by='board_id') \
            .project(
                ne.all(),
                good_board_fielddate=ne.custom(lambda cdt, udt: udt if cdt is None or (udt is not None and cdt < udt) else cdt, 'card_created_at_date_for_3th_card',
                                               'unlock_date').with_type(qt.Optional[qt.String])
            )

    @require('Corgie.corgie')
    def put_corgie(self, streams):
        return streams['Corgie.corgie'] \
            .put('//home/collections/analytics/backups/corgie/corgie_v1_latest')

    @require(
        'Corgie.corgie',
        'PrepCards.organic_cards_without_spam',
    )
    def corgie_cards(self, streams):
        boards = streams['Corgie.corgie'] \
            .project(
                'board_features_corgie_v2',
                'board_features_corgie_v3',
                card_board_id='board_id',
            )

        return streams['PrepCards.organic_cards_without_spam'] \
            .join(
                boards,
                type='inner',
                by='card_board_id',
                allow_undefined_keys=False
            )

    @require('Corgie.corgie_cards')
    def put_corgie_cards(self, streams):
        return streams['Corgie.corgie_cards'] \
            .put('//home/collections/analytics/backups/corgie/all_cards_from_corgie_v1_latest')

    @require('Corgie.corgie', 'PrepUsers.organics')
    def unload_corgie_daily(self, streams):
        # Корги выгружаем минимум за 3 дня от даты запуска крона. Для того, чтобы база наверняка была свежая и таблицы бы были.
        # Для того, чтобы крон хитмана запускался только 1 раз при первом создании таблички https://hitman.yandex-team.ru/projects/collections_stuff/corgie_v2_toloka/
        for table_stream in split_by_dates(
            streams['Corgie.corgie'],
            [x.strftime(DATE_FORMAT) for x in date_range(self.date, self.dateend, delta=2)],
            '//home/collections/analytics/backups/corgie/corgie_v1_daily/',
            date_field='good_board_fielddate',
        ):
            yield table_stream

    def corgie_features_coverage(self, stream, feature_name, stat_report):
        date_start = self.date.strftime(DATE_FORMAT)
        date_end = self.dateend.strftime(DATE_FORMAT)

        schema = {
            'good_board_fielddate': str,
        }
        schema[feature_name] = str

        return stream \
            .filter(
                nf.and_(
                    qf.nonzero(feature_name),
                    nf.custom(lambda dt: date_start <= dt <= date_end, 'good_board_fielddate')
                )
            ) \
            .map(with_hints(output_schema=schema)(
                AddTotalsMapper(['good_board_fielddate', feature_name], ['good_board_fielddate'], total_value='total_db')
            )) \
            .groupby('good_board_fielddate', feature_name) \
            .aggregate(
                count=na.count()
            ) \
            .project(
                'count',
                fielddate='good_board_fielddate',
                verdict=ne.custom(lambda x: str(x) + '_db', feature_name).with_type(str),
                reason=ne.const('_total_').with_type(str),
            ) \
            .publish(self.get_statface_report(stat_report), allow_change_job=True)

    @require('Corgie.corgie')
    def corgie_v2_db_stats(self, streams):
        return self.corgie_features_coverage(
            streams['Corgie.corgie'],
            'board_features_corgie_v2',
            'Collections/Metrics/kok_and_corgie/corgie_v2',
        )

    @require('Corgie.corgie')
    def corgie_v3_db_stats(self, streams):
        return self.corgie_features_coverage(
            streams['Corgie.corgie'],
            'board_features_corgie_v3',
            'Collections/Metrics/kok_and_corgie/corgie_v3',
        )

    @require('PrepBoards.with_organic_flag', 'Corgie.corgie')
    def prep_daily_corgie(self, streams):
        REPORT_KEY_FIELDS = ['fielddate', 'user_type', 'corgie_v1', 'corgie_v2', 'corgie_v3']

        return streams['PrepBoards.with_organic_flag'] \
            .filter(nf.not_(nf.equals('board_is_private', True))) \
            .join(streams['Corgie.corgie'].project('board_id', is_corgie=ne.const(True)), by='board_id', type='left') \
            .project(
                'user_type',
                fielddate='board_created_at_date',
                corgie_v1=ne.custom(lambda x: 'ok' if x else 'not_ok', 'is_corgie').with_type(str),
                corgie_v2=ne.custom(lambda x: x if x else 'empty', 'board_features_corgie_v2').with_type(str),
                corgie_v3=ne.custom(lambda x: x if x else 'empty', 'board_features_corgie_v3').with_type(str)
            ) \
            .map(with_hints(output_schema=extended_schema())(
                AddTotalsMapper(REPORT_KEY_FIELDS, ['fielddate'])
            )) \
            .groupby(*REPORT_KEY_FIELDS) \
            .aggregate(boards=na.count())

    @require('Corgie.prep_daily_corgie')
    def corgie_accum_publish(self, streams):

        return streams['Corgie.prep_daily_corgie'] \
            .groupby('user_type', 'corgie_v1', 'corgie_v2', 'corgie_v3') \
            .sort('fielddate') \
            .reduce(with_hints(output_schema=extended_schema())(AccumReducer(self.date, self.dateend))) \
            .publish(self.get_statface_report('Collections/Metrics/kok_and_corgie/CorgieAccumV1'), allow_change_job=True)

    @require('Corgie.prep_daily_corgie')
    def corgie_daily_publish(self, streams):
        streams['Corgie.prep_daily_corgie'] \
            .publish(self.get_statface_report('Collections/Metrics/kok_and_corgie/CorgieDailyV1'), allow_change_job=True)
