# coding: utf-8

from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from analytics.plotter_lib.utils import split_by_dates, date_range
from nile.api.v1 import (
    extractors as ne,
    aggregators as na,
    filters as nf,
)

from qb2.api.v1 import (
    filters as qf
)


class PrepCards(Plot):
    @require('Cards.parsed')
    def without_bookmarks(self, streams):
        stream = streams['Cards.parsed']

        return stream.filter(nf.not_(nf.equals('card_is_browser_bookmark', True)))

    @require('PrepUsers.with_organic_flag', 'PrepCards.without_bookmarks')
    def with_organic_flag(self, streams):
        """Карточки без закладок и импорта"""

        users = streams['PrepUsers.with_organic_flag'] \
            .project(
                'user_id',
                'user_login',
                'user_type',
                'user_is_partner',
                'user_created_date',
                'user_company_id',
                'user_ban',
                owner_is_organic='user_is_organic',
                content_owner_puid='user_puid',
            )

        return streams['PrepCards.without_bookmarks'] \
            .join(
                users,
                type='inner',
                by_left='card_owner_id',
                by_right='user_id',
            ) \
            .project(
                ne.all(['owner_is_organic', 'user_is_partner']),
                card_owner_is_organic=ne.custom(lambda x: True if x else False, 'owner_is_organic').with_type(bool),
                card_owner_is_partner=ne.custom(lambda x: True if x else False, 'user_is_partner').with_type(bool),
            )

    @require('PrepCards.with_organic_flag')
    def organic_cards(self, streams):
        cards = streams['PrepCards.with_organic_flag']

        return cards.filter(nf.and_(nf.equals('card_owner_is_organic', True), nf.not_(qf.nonzero('user_ban'))))

    @require('PrepCards.organic_cards', 'TagTable.parsed')
    def organic_cards_without_spam(self, streams):
        organic_cards = streams['PrepCards.organic_cards']
        tag_table = streams['TagTable.parsed']

        public_cards, private_cards = organic_cards.filter(nf.not_(nf.equals('card_is_banned', True))) \
            .split(nf.equals('card_is_private', True))

        public_cards_without_spam = public_cards.join(
            tag_table.filter(nf.custom(lambda prob: prob > 0.9, 'board_adv_prob')),
            by_left='card_board_id',
            by_right='board_id',
            type='left_only',
            allow_undefined_keys=False
        )

        return public_cards_without_spam.concat(private_cards)

    @require('PrepCards.with_organic_flag')
    def board_first_cards(self, streams):
        return streams['PrepCards.with_organic_flag'] \
            .groupby('card_board_id') \
            .top(1, by='card_created_at_timestamp', mode='min')

    @require('PrepCards.organic_cards_without_spam')
    def put_organic_cards_without_spam(self, streams):
        stream = streams['PrepCards.organic_cards_without_spam']

        return stream.put('//home/collections/analytics/backups/cards_without_spam/latest')

    @require('Cards.parsed')
    def bookmarks(self, streams):
        bookmarks = streams['Cards.parsed'].filter(nf.equals('card_is_browser_bookmark', True))

        return bookmarks.project(fielddate='card_created_at_date') \
            .groupby('fielddate') \
            .aggregate(bookmarks=na.count()) \
            .publish(self.get_statface_report('Collections/Metrics/bookmarks_by_day'), allow_change_job=True)

    @require('PrepCards.organic_cards_without_spam', 'PrepUsers.organics', 'PrepBoards.organic_boards_without_spam')
    def put_products_without_spam(self, streams):
        boards = streams['PrepBoards.organic_boards_without_spam'].project('board_slug', card_board_id='board_id')

        products = streams['PrepCards.organic_cards_without_spam'] \
            .filter(nf.equals('card_meta_info_type', 'product')) \
            .join(boards, type='inner', by='card_board_id') \
            .project(
                'card_id',
                'card_created_at_date',
                'card_is_private',
                'card_description',
                'card_owner_id',
                'card_avatars_url',
                product_title='card_meta_info_data_title',
                card_url=ne.custom(lambda x: 'https://yandex.ru/collections/card/{}/'.format(x), 'card_id').with_type(str),
                board_url=ne.custom(lambda x, y: 'https://yandex.ru/collections/user/{}/{}'.format(x, y), 'user_login', 'board_slug').with_type(str)
            )

        return split_by_dates(
            products,
            [x.strftime(DATE_FORMAT) for x in date_range(self.date, self.dateend, delta=2)],
            '//home/collections/analytics/backups/product/daily/',
            date_field='card_created_at_date',
        )

    @require('PrepCards.with_organic_flag')
    def new_cards_with_organic_flag(self, streams):
        date_start = self.date.strftime(DATE_FORMAT)
        date_end = self.dateend.strftime(DATE_FORMAT)

        return streams['PrepCards.with_organic_flag'] \
            .filter(nf.custom(lambda dt: date_start <= dt <= date_end, 'card_created_at_date'))
