# coding: utf-8

import json

from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from analytics.plotter_lib.utils import fromtimestamp
from nile.api.v1 import (
    Record,
    with_hints,
    extended_schema,
    extractors as ne,
    filters as nf
)

from qb2.api.v1.resources import resource, get
from qb2.api.v1 import (
    typing as qt
)


def convert_convert_action_to_redir_path(action):
    if action == 'UA_POST':
        return 'finish.comment'
    elif action == 'UA_REACTION':
        return 'finish.comment.reaction'
    elif action == 'UA_REPLY':
        return 'finish.comment.reply'


class ParseCmntAPIAccessLog(object):

    def __init__(self, uatraits_resource):
        self.uatraits_resource = uatraits_resource

    def __call__(self, recs):
        detector = get(self.uatraits_resource)
        for rec in recs:
            data = rec.to_dict()

            try:
                data['fielddate'] = fromtimestamp(float(rec.get('event_timestamp', 0)) / 1000000).strftime(DATE_FORMAT)
            except:
                continue

            try:
                stats = json.loads(rec.get('Stats'))
                data['Stats_loc'] = str(stats.get('loc'))
            except Exception:
                pass

            try:
                parsed_os = detector.detect(rec.UserAgent)
                if parsed_os.get('isTablet'):
                    data['device'] = 'tablet'
                elif parsed_os.get('isMobile'):
                    data['device'] = 'mobile'
                else:
                    data['device'] = 'desktop'
            except:
                data['device'] = None

            yield Record.from_dict(data)


class CmntAPIAccessLog(Plot):

    @require('cmnt-api-access-log')
    def raw(self):
        return {}

    @require('CmntAPIAccessLog.raw')
    def parsed(self, streams):
        stream = streams['CmntAPIAccessLog.raw']

        return stream.map(
            with_hints(output_schema=extended_schema(fielddate=str, Stats_loc=qt.Optional[qt.String], device=qt.Optional[qt.String]))(
                ParseCmntAPIAccessLog(resource('UATraits'))
            )
        )

    @require('CmntAPIAccessLog.parsed')
    def get_collections(self, streams):
        return streams['CmntAPIAccessLog.parsed'].filter(nf.equals('ServiceSlug', 'collections'))

    @require('CmntAPIAccessLog.get_collections')
    def get_collections_one_day(self, streams):
        dateend = self.dateend.strftime(DATE_FORMAT)

        return streams['CmntAPIAccessLog.get_collections'].filter(nf.equals('fielddate', dateend))

    @require('CmntAPIAccessLog.get_collections')
    def convert_to_redir_format(self, streams):
        cmnt_log = streams['CmntAPIAccessLog.get_collections']

        return cmnt_log \
            .filter(
                nf.and_(
                    nf.equals('Success', True),
                    nf.custom(lambda x: x in ('UA_POST', 'UA_REACTION', 'UA_REPLY'), 'UserAction'),
                    nf.custom(lambda x: bool(x), 'YandexUid')
                )
            ) \
            .project(
                fielddate=ne.custom(lambda x: x, 'fielddate').with_type(qt.Optional[qt.String]),
                iso_eventtime=ne.custom(lambda x: x, 'iso_eventtime').with_type(qt.Optional[qt.String]),
                yandexuid=ne.custom(lambda x: x, 'YandexUid').with_type(qt.Optional[qt.String]),
                puid=ne.custom(lambda x: str(x) if x else None, 'Puid').with_type(qt.Optional[qt.String]),
                ui=ne.custom(lambda x: x, 'device').with_type(qt.Optional[qt.String]),
                page_name=ne.const('undefined').with_type(qt.Optional[qt.String]),
                path=ne.custom(convert_convert_action_to_redir_path, 'UserAction').with_type(qt.Optional[qt.String]),
                timestamp=ne.custom(lambda x: int(x / 1000000), 'event_timestamp').with_type(qt.Optional[qt.Integer]),
                entity_id=ne.custom(lambda x: x[0] if x and len(x) > 0 else None, 'EntityId').with_type(qt.Optional[qt.String]),
            )
