# coding: utf-8


from time_spent import filter_session
from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from analytics.plotter_lib.utils import AddTotalsMapper, split_sessions
from nile.api.v1 import (
    Record,
    with_hints,
    filters as nf,
    extractors as ne,
    aggregators as na,
    extended_schema
)


class PrepPushLog(Plot):
    @require('push-log')
    def raw(self):
        return {}

    @require('PrepPushLog.raw')
    def get_collections(self, streams):
        return streams['PrepPushLog.raw'] \
            .filter(nf.equals('project', 'podb'))

    @require('PrepPushLog.get_collections')
    def get_collections_shown_one_day(self, streams):
        date_str = self.date.strftime(DATE_FORMAT)
        dateend_str = self.dateend.strftime(DATE_FORMAT)
        return streams['PrepPushLog.get_collections'] \
            .filter(
                nf.and_(
                    nf.custom(lambda x: True if x and 'Shown' in x else False, 'push_metrika_events'),
                    nf.custom(lambda x: date_str <= x <= dateend_str, 'log_date')
                )
            )


class SupApps(Plot):
    @require('//home/search-functionality/sup/backup/sup-prod.n.yandex-team.ru/latest')
    def raw(self, streams):
        return streams['//home/search-functionality/sup/backup/sup-prod.n.yandex-team.ru/latest'].project(ne.all())

    @require('SupApps.raw')
    def device_id_puid(self, streams):
        return streams['SupApps.raw'] \
            .project(device_id='deviceId', puid='uid') \
            .filter(
                nf.and_(
                    nf.custom(lambda x: True if x else False, 'device_id'),
                    nf.custom(lambda x: True if x else False, 'puid')
                )
            )


@with_hints(
    output_schema=dict(
        fielddate=str,
        push_shown_users=int,
        collections_users=int,
        users_with_good_session=int,
        platform=str
    )
)
def visits_reducer(groups):
    for key, recs in groups:
        sessions = split_sessions(recs)
        has_access = False
        has_good_session = False

        for session in sessions:
            filtered_session = filter_session(session, 'has_access')

            has_access = True if has_access or filtered_session else False

            session_duration = int(filtered_session[-1].timestamp) - int(filtered_session[0].timestamp) if filtered_session else 0
            # Длина сессии(сек.) должна быть меньше чем секунд в дне
            if 120 <= session_duration <= 60 * 60 * 24:
                has_good_session = True
                break

        yield Record(
            fielddate=key.fielddate,
            push_shown_users=1,
            collections_users=int(has_access),
            users_with_good_session=int(has_good_session),
            platform=key.platform
        )


class PushMetrics(Plot):
    @require('SupApps.device_id_puid', 'PrepPushLog.get_collections_shown_one_day', 'CollectionsRedirLog.parsed')
    def push_metrics(self, streams):
        device_id_puid = streams['SupApps.device_id_puid']
        push_shown_ts = streams['PrepPushLog.get_collections_shown_one_day'] \
            .project(
                'device_id',
                'platform',
                fielddate='log_date',
                shown_ts=ne.custom(lambda x: int(x.get('timestamp', 0)), 'push_metrika_events').with_type(int)
            ) \
            .join(device_id_puid, by='device_id', type='inner') \
            .project('puid', 'shown_ts', 'platform', 'fielddate') \
            .groupby('puid') \
            .top(1, by='shown_ts', mode='min')

        redir = streams['CollectionsRedirLog.parsed'] \
            .project('puid', 'timestamp', 'path')

        add_total_fields = ['platform']
        add_total_pass_fields = ['fielddate', 'push_shown_users', 'collections_users', 'users_with_good_session']

        return push_shown_ts \
            .join(redir, by='puid', type='left') \
            .project(
                ne.all('timestamp'),
                timestamp=ne.custom(lambda x, y: x if x else y, 'timestamp', 'shown_ts').with_type(int)
            ) \
            .filter(nf.custom(lambda x, y: x <= y, 'shown_ts', 'timestamp')) \
            .groupby('fielddate', 'puid', 'platform') \
            .sort('timestamp') \
            .reduce(visits_reducer) \
            .map(
                with_hints(output_schema=extended_schema())(
                    AddTotalsMapper(add_total_fields + add_total_pass_fields, add_total_pass_fields)
                )
            ) \
            .groupby('fielddate', 'platform') \
            .aggregate(
                push_shown_users=na.sum('push_shown_users'),
                collections_users=na.sum('collections_users'),
                users_with_good_session=na.sum('users_with_good_session')
            ) \
            .publish(self.get_statface_report('Collections/Metrics/Pushs/PushUsers'), allow_change_job=True)
