
import itertools

from analytics.plotter_lib.plotter import Plot, require
from nile.api.v1 import (
    with_hints,
    extended_schema,
    extractors as ne,
    aggregators as na,
    filters as nf
)


@with_hints(output_schema=extended_schema())
def add_totals(stream):
    for r in stream:
        for comb in itertools.product(
            (r.ui, '_total_'),
            (r.page_name, '_total_'),
            (r.traffic_source, '_total_'),
        ):
            ui, page_name, traffic_source = comb
            yield dict(
                fielddate=r.fielddate,
                yandexuid=r.yandexuid,
                ui=ui,
                page_name=page_name,
                traffic_source=traffic_source,
            )


class SessionsByPageName(Plot):
    @require('CollectionsRedirLog.parsed')
    def calculate(self, streams):
        return streams['CollectionsRedirLog.parsed'] \
            .filter(
                nf.and_(
                    nf.equals('path', 'start.session'),
                    nf.custom(lambda page_name: False if page_name is None or page_name == '' else True, 'page_name')
                )
            ) \
            .project(
                'fielddate',
                ui=ne.custom(lambda x: 'empty' if x is None else x, 'ui').with_type(str),
                page_name=ne.custom(
                    lambda x, y: 'browser_favorites' if y == 'bro_desktop' else x, 'page_name', 'traffic_source'
                ).with_type(str)
            ) \
            .groupby('fielddate', 'ui', 'page_name') \
            .aggregate(sessions_count=na.count()) \
            .publish(self.get_statface_report('Collections/Metrics/sessions_by_page_name'), allow_change_job=True)


class SessionsByTrafficSourcePageName(Plot):
    @require('CollectionsRedirLog.parsed')
    def calculate(self, streams):
        return streams['CollectionsRedirLog.parsed'] \
            .filter(
                nf.and_(
                    nf.equals('path', 'start.session'),
                    nf.custom(lambda page_name: False if page_name is None or page_name == '' else True, 'page_name')
                )
            ) \
            .project(
                'fielddate',
                'yandexuid',
                'traffic_source',
                ui=ne.custom(lambda x: 'empty' if x is None else x, 'ui').with_type(str),
                page_name=ne.custom(
                    lambda x, y: 'browser_favorites' if y == 'bro_desktop' else x, 'page_name', 'traffic_source'
                ).with_type(str),
            ) \
            .map(add_totals) \
            .groupby('fielddate', 'ui', 'page_name', 'traffic_source') \
            .aggregate(
                sessions_count=na.count(),
                users_count=na.count_distinct('yandexuid')
            ) \
            .publish(self.get_statface_report('Collections/Metrics/sessions/sessions_by_trafficsource_pagename'), allow_change_job=True)
