# coding: utf-8

from collections import defaultdict
from copy import deepcopy

from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from analytics.plotter_lib.utils import fromtimestamp, parse_vars
from analytics.collections.plotter_collections.plots.utils import get_request_ui

from nile.api.v1 import (
    Record,
)
from qb2.api.v1 import (
    typing as qt
)


def extract_images_kokoko_shows_clicks(r):
    base_data = {
        'service': 'images',
        'page_name': 'serp',
        'location': 'kokoko',
        'ui': get_request_ui(r),
        'timestamp': int(r.Timestamp),
        'fielddate': fromtimestamp(int(r.Timestamp)).strftime(DATE_FORMAT),
        'search_query': r.Query,
        'reqid': r.ReqId,
        'yandexuid': r.UID,
    }
    if r.PassportUID:
        base_data['puid'] = 'p' + r.PassportUID

    clicked_boards = defaultdict(int)

    # touch or desktop after REVIEW:1142558
    for click in r.GetClicks():
        if click.ConvertedPath.endswith('/serp/collections/click'):
            click_vars = parse_vars(click.GetVars())
            clicked_boards[click_vars.get('-board_id')] += 1

    # old desktop:
    for click in r.GetYandexTechEvents():
        if click.Path.endswith('.471.2177.882'):  # /serp/collections/click
            click_vars = parse_vars(click.Vars)
            clicked_boards[click_vars.get('-board_id')] += 1

    for b in r.GetBSBlocks():
        if b.Path.endswith('/serp/collections'):
            collections_vars = {k: v for k, v in b.GetVars()}
            if '-boards' in collections_vars:
                # С 14:48 5 декабря 2019 г. IMAGESUI-11398: Логировать показанные board_id в Кококо (desktop, touch)
                boards_shown = collections_vars['-boards'].split(',')
                for idx, board_id in enumerate(boards_shown):
                    board_data = deepcopy(base_data)
                    board_data['content_type'] = 'board'
                    board_data['board_id'] = board_id
                    board_data['block_position'] = int(collections_vars['-row'])
                    board_data['content_position'] = idx
                    if board_id in clicked_boards:
                        board_data['clicks'] = clicked_boards[board_id]
                    yield board_data
            else:
                # Раньше не логировались показанные коллекции, пишем в выжимки только кликнутые коллекции
                for board_id, clicks_count in clicked_boards.items():
                    board_data = deepcopy(base_data)
                    board_data['content_type'] = 'board'
                    board_data['board_id'] = board_id
                    board_data['clicks'] = clicks_count
                    yield board_data
                return


class ImagesUserSessionsSearch(Plot):
    def __init__(self, *args, **kwargs):
        super(ImagesUserSessionsSearch, self).__init__(*args, **kwargs)
        # Удаляем job из класса. Pickle не умеет nile job'ы прокидывать в yt операции
        self.job = None

    @require('user_sessions', layer='sessions', schema={
        'service': qt.String,
        'page_name': qt.String,
        'location': qt.String,
        'ui': qt.String,
        'timestamp': qt.Int32,
        'fielddate': qt.String,
        'search_query': qt.Optional[qt.String],
        'reqid': qt.Optional[qt.String],
        'yandexuid': qt.Optional[qt.String],
        'puid': qt.Optional[qt.String],
        'content_type': qt.String,
        'board_id': qt.Optional[qt.String],
        'block_position': qt.Optional[qt.Int32],
        'content_position': qt.Optional[qt.Int32],
        'clicks': qt.Optional[qt.Int32],
    })
    def get_kokoko_shows(self, requests_container):
        for r in requests_container.GetRequests():
            if not r.IsA('TImagesRequestProperties'):
                continue

            for record in extract_images_kokoko_shows_clicks(r):
                yield Record.from_dict(record)

    @require('ImagesUserSessionsSearch.get_kokoko_shows', layer='sessions')
    def collections_shows(self, streams):
        return streams['ImagesUserSessionsSearch.get_kokoko_shows'] \
            .checkpoint(self.get_checkpoint_name('images_kokoko_shows'))
