# coding: utf-8

from copy import deepcopy
from collections import Counter

from nile.api.v1 import (
    Record,
)
from qb2.api.v1 import (
    typing as qt
)

from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from analytics.plotter_lib.utils import fromtimestamp
from analytics.collections.plotter_collections.plots.utils import get_request_ui


def extract_morda_informer_collectons_shows(r):  # noqa
    base_data = {
        'service': 'morda',
        'page_name': 'morda',
        'location': 'informer',
        'ui': get_request_ui(r),
        'timestamp': int(r.Timestamp),
        'fielddate': fromtimestamp(int(r.Timestamp)).strftime(DATE_FORMAT),
        'reqid': r.ReqId,
        'yandexuid': r.UID,
    }
    if r.PassportUID:
        base_data['puid'] = 'p' + r.PassportUID

    for b in r.GetMainBlocks():
        for res in b.GetChildren():
            if not res:
                continue

            if not hasattr(res, 'Path') or (res.Path != 'collections' and res.Path != 'div_collections'):
                continue

            is_real_show = False
            board_shows, card_shows = {}, {}
            board_clicks, card_clicks = Counter(), Counter()

            for e in res.GetOwnEvents():
                if 'collections.realshow' in e.Path:
                    is_real_show = True
                    continue

                if e.IsA('TClick'):
                    if 'collections.channel' in e.Path or 'div_collections.board' in e.Path:
                        board_clicks[e.Path] += 1
                    elif 'collections.feed' in e.Path:
                        card_clicks[e.Path] += 1
                elif e.IsA('TPortalShow'):
                    if 'collections.channel' in e.Path or 'div_collections.board' in e.Path:
                        board_shows[e.Path] = e.BlockId
                    elif 'collections.feed' in e.Path:
                        card_shows[e.Path] = e.BlockId

            if not is_real_show:
                continue

            base_data['block_position'] = int(res.Position)
            for idx, (path, board_id) in enumerate(sorted(board_shows.items())):
                board_data = deepcopy(base_data)
                board_data['content_type'] = 'board'
                board_data['board_id'] = str(board_id)
                board_data['content_position'] = idx
                if path in board_clicks:
                    board_data['clicks'] = board_clicks[path]
                yield board_data

            for idx, (path, card_id) in enumerate(sorted(card_shows.items())):
                card_data = deepcopy(base_data)
                card_data['content_type'] = 'card'
                card_data['card_id'] = str(card_id)
                card_data['content_position'] = idx
                if path in card_clicks:
                    card_data['clicks'] = card_clicks[path]
                yield card_data


class MordaUserSessionsSearch(Plot):
    def __init__(self, *args, **kwargs):
        super(MordaUserSessionsSearch, self).__init__(*args, **kwargs)
        # Удаляем job из класса. Pickle не умеет nile job'ы прокидывать в yt операции
        self.job = None

    @require('user_sessions', layer='sessions', schema={
        'service': qt.String,
        'page_name': qt.String,
        'location': qt.String,
        'ui': qt.String,
        'timestamp': qt.Int32,
        'fielddate': qt.String,
        'reqid': qt.Optional[qt.String],
        'yandexuid': qt.Optional[qt.String],
        'puid': qt.Optional[qt.String],
        'content_type': qt.String,
        'board_id': qt.Optional[qt.String],
        'card_id': qt.Optional[qt.String],
        'block_position': qt.Optional[qt.Int64],
        'content_position': qt.Optional[qt.Int32],
        'clicks': qt.Optional[qt.Int32],
    })
    def get_informer_shows(self, requests_container):
        for r in requests_container.GetRequests():
            if r.ServiceType != 'portal':
                continue

            for record in extract_morda_informer_collectons_shows(r):
                yield Record.from_dict(record)

    @require('MordaUserSessionsSearch.get_informer_shows', layer='sessions')
    def collections_shows(self, streams):
        return streams['MordaUserSessionsSearch.get_informer_shows'] \
            .checkpoint(self.get_checkpoint_name('morda_collections_shows'))
