# coding: utf-8

from copy import deepcopy
from collections import defaultdict

from nile.api.v1 import (
    Record,
    extractors as ne,
)
from qb2.api.v1 import (
    filters as qf,
    typing as qt,
)

from analytics.plotter_lib.plotter import Plot, require, DATE_FORMAT
from analytics.plotter_lib.utils import fromtimestamp
from analytics.collections.plotter_collections.plots.utils import (
    get_block_clicks_count,
    get_baobab_block_index,
    get_parent_block_by_name,
    get_request_ui,
    get_collections_page,
    get_card_id_from_url,
    get_user_login_board_slug_from_url,
)


def get_result_pos(block):
    result_block = get_parent_block_by_name(block, '$result')
    if result_block and 'pos' in result_block.attrs:
        return int(result_block.attrs['pos'])
    return None


def extract_serp_kokos_data(r):  # noqa
    import tamus
    import baobab

    base_data = {
        'service': 'web',
        'page_name': 'serp',
        'location': 'kokos',
        'ui': get_request_ui(r),
        'timestamp': int(r.Timestamp),
        'fielddate': fromtimestamp(int(r.Timestamp)).strftime(DATE_FORMAT),
        'search_query': r.Query,
        'reqid': r.ReqId,
        'yandexuid': r.UID,
    }
    if r.PassportUID:
        base_data['puid'] = 'p' + r.PassportUID

    joiners = r.BaobabAllTrees()
    if not joiners:
        return

    rules = {
        'organic_main_result': '$main / $result [@type = "organic"]',
        'collections_snippet': '#organic_main_result / collections',  # TODO: переписать на tamus как только будет поддержка "@features == ['collections-snippet']"

        'desktop': '$page [@ui = "desktop"]',
        'mobile': '$page [@ui = "mobapp" or @ui = "touch"]',  # TODO: проверить что действительно mobapp
        'wizard_collections': '$result [@type = "wizard" and @wizard_name = "collections"]',

        'wizard_collections_list': '#wizard_collections [@subtype = "list"]',
        'wizard_collections_list__board': '#wizard_collections_list / collection-list // collection',

        'desktop_wizard_collections_user': '#desktop // #wizard_collections [@subtype = "user"]',
        'desktop_wizard_collections_user__board': '#desktop_wizard_collections_user / (collections-user or collections-list) / scroller / collection',

        'mobile_wizard_collections_user': '#mobile // #wizard_collections [@subtype = "user"]',
        'mobile_wizard_collections_user__board': '#mobile_wizard_collections_user // scroller / collection-card',

        'wizard_collections_viewer': '$subresult / collection-viewer-root / collection-viewer',
        'wizard_collections_viewer__content': '#wizard_collections_viewer / preview / collection-content',
        'wizard_collections_viewer__sidebar': '#wizard_collections_viewer / sidebar',
    }
    marks = tamus.check_rules_multiple_joiners_merged(rules, joiners)

    # переходы по карточкам-ссылкам из просмотрщика на десктопе на серпе https://nda.ya.ru/3Vsvxi
    card_viewer_greenurls = defaultdict(int)
    for block in marks.get_blocks('wizard_collections_viewer__content'):
        inner_block = block.first_child
        if inner_block and inner_block.name == 'link':
            card_id = str(block.attrs.get('cardId'))
            greenurls_count = get_block_clicks_count(inner_block, joiners)
            if greenurls_count > 0:
                card_viewer_greenurls[card_id] = greenurls_count

    # карточки, загруженные в просмотрщик на десктопе
    for block in marks.get_blocks('wizard_collections_viewer__sidebar'):
        card_data = deepcopy(base_data)
        card_data['location'] = 'serpviewer'
        card_data['content_type'] = 'card'
        card_data['card_id'] = str(block.attrs.get('card_id'))
        card_data['content_position'] = get_baobab_block_index(block, same_name=True)

        if card_viewer_greenurls.get(card_data['card_id'], 0) > 0:
            card_data['greenurls'] = card_viewer_greenurls.get(card_data['card_id'], 0)

        for inner_block in baobab.common.bfs_iterator(block):
            if inner_block.name == 'link' and inner_block.prev_sibling and inner_block.prev_sibling.name == 'link':
                # вторая ссылка в сайдбаре — это гринурл на сайт TODO: refactor after PODB-16071
                clicks_count = get_block_clicks_count(inner_block, joiners)
                if clicks_count > 0:
                    card_data['greenurls'] = clicks_count + card_data.get('greenurls', 0)
                break

        for inner_block in baobab.common.bfs_iterator(block):
            is_greenurl = False

            # вторая ссылка в сайдбаре с пустыми attrs — это гринурл на сайт до 2020-01-27
            if inner_block.name == 'link' and not inner_block.attrs and inner_block.prev_sibling and inner_block.prev_sibling.name == 'link' and not inner_block.prev_sibling.prev_sibling:
                is_greenurl = True

            # гринурл в сайдбаре отмечен в attrs после PODB-16071
            if inner_block.name == 'link' and inner_block.attrs and inner_block.attrs.get('link_external') is True:
                is_greenurl = True

            if is_greenurl:
                greenurls_count = get_block_clicks_count(inner_block, joiners)
                if greenurls_count > 0:
                    card_data['greenurls'] = card_data.get('greenurls', 0) + greenurls_count
                    break

        yield card_data

    # спецсниппет коллекций на серпе
    for block in marks.get_blocks('collections_snippet'):
        board_data = deepcopy(base_data)
        board_data['location'] = 'collections-snippet'
        board_data['content_type'] = 'board'
        board_data['board_id'] = str(block.attrs.get('id'))
        board_data['block_position'] = get_result_pos(block)

        clicks_count = sum([get_block_clicks_count(x, joiners)
                            for x in baobab.common.bfs_iterator(block)])
        if clicks_count > 0:
            board_data['clicks'] = clicks_count
        yield board_data

    # показы и клики по органическим документам про Коллекции: только карточки, борды
    for block in marks.get_blocks('organic_main_result'):
        url = block.attrs.get('documentUrl')
        if not url:
            continue
        collections_page = get_collections_page(url)
        if not collections_page or collections_page not in ['board', 'card']:
            continue

        item_data = deepcopy(base_data)
        item_data['location'] = 'organic'
        item_data['block_position'] = get_result_pos(block)

        clicks_count = sum([get_block_clicks_count(x, joiners)
                            for x in baobab.common.bfs_iterator(block)])
        if clicks_count > 0:
            item_data['clicks'] = clicks_count

        item_data['content_type'] = collections_page
        if collections_page == 'card':
            item_data['card_id'] = get_card_id_from_url(url)
        else:
            user_login, board_slug = get_user_login_board_slug_from_url(url)
            item_data['user_login'] = user_login
            item_data['board_slug'] = board_slug

        yield item_data

    # однорядный колдунщик subtype = user, десктоп
    for block in marks.get_blocks('desktop_wizard_collections_user__board'):
        board_data = deepcopy(base_data)
        board_data['content_type'] = 'board'
        board_data['board_id'] = str(block.attrs.get('id'))
        board_data['content_position'] = get_baobab_block_index(block, same_name=True)
        board_data['block_position'] = get_result_pos(block)

        clicks_count = sum([get_block_clicks_count(x, joiners)
                            for x in baobab.common.bfs_iterator(block)
                            if x.name in ['title', 'preview']])
        if clicks_count > 0:
            board_data['clicks'] = clicks_count

        yield board_data

    # однорядный колдунщик subtype = user, тачи
    for block in marks.get_blocks('mobile_wizard_collections_user__board'):
        board_data = deepcopy(base_data)
        board_data['content_type'] = 'board'
        board_data['board_id'] = str(block.attrs.get('id'))
        board_data['content_position'] = get_baobab_block_index(block, same_name=True)
        board_data['block_position'] = get_result_pos(block)

        clicks_count = sum([get_block_clicks_count(x, joiners)
                            for x in baobab.common.bfs_iterator(block)])
        if clicks_count > 0:
            board_data['clicks'] = clicks_count

        yield board_data

    # многорядный колдунщик subtype = list, учитываем только показы коллекций, карточки не считаем
    for block in marks.get_blocks('wizard_collections_list__board'):
        board_data = deepcopy(base_data)
        board_data['content_type'] = 'board'
        board_data['board_id'] = str(block.attrs.get('id'))
        board_data['content_position'] = get_baobab_block_index(block, same_name=True)
        board_data['block_position'] = get_result_pos(block)

        clicks_count = sum([get_block_clicks_count(x, joiners)
                            for x in baobab.common.bfs_iterator(block)
                            if x.name == 'title'])
        if clicks_count > 0:
            board_data['clicks'] = clicks_count
        yield board_data


class WebUserSessionsSearch(Plot):
    def __init__(self, *args, **kwargs):
        super(WebUserSessionsSearch, self).__init__(*args, **kwargs)
        # Удаляем job из класса. Pickle не умеет nile job'ы прокидывать в yt операции
        self.job = None

    @require('user_sessions', layer='sessions', schema={
        'service': qt.String,
        'page_name': qt.String,
        'location': qt.String,
        'ui': qt.String,
        'timestamp': qt.Int32,
        'fielddate': qt.String,
        'search_query': qt.Optional[qt.String],
        'reqid': qt.Optional[qt.String],
        'yandexuid': qt.Optional[qt.String],
        'puid': qt.Optional[qt.String],
        'content_type': qt.String,
        'board_id': qt.Optional[qt.String],
        'card_id': qt.Optional[qt.String],
        'block_position': qt.Optional[qt.Int32],
        'content_position': qt.Optional[qt.Int32],
        'clicks': qt.Optional[qt.Int32],
        'greenurls': qt.Optional[qt.Int32],
        'user_login': qt.Optional[qt.String],
        'board_slug': qt.Optional[qt.String],
    })
    def get_collections_shows(self, requests_container):
        for r in requests_container.GetRequests():
            if not r.IsA('TWebRequestProperties'):
                continue
            if not r.IsA('TBaobabProperties'):
                continue

            for record in extract_serp_kokos_data(r):
                yield Record.from_dict(record)

    @require(
        'WebUserSessionsSearch.get_collections_shows',
        'PrepBoards.expanded_slugs',
        layer='sessions'
    )
    def joined_with_board_id(self, streams):
        other_records, records_board_slugs = streams['WebUserSessionsSearch.get_collections_shows'] \
            .split(qf.nonzero('board_slug'))

        return records_board_slugs \
            .project(ne.all('board_id')) \
            .join(
                streams['PrepBoards.expanded_slugs'].project('user_login', 'board_slug', 'board_id'),
                type='inner',
                by=['user_login', 'board_slug']
            ) \
            .concat(other_records) \
            .project(ne.all(['user_login', 'board_slug']))

    @require('WebUserSessionsSearch.joined_with_board_id', layer='sessions')
    def collections_shows(self, streams):
        return streams['WebUserSessionsSearch.joined_with_board_id'] \
            .checkpoint(self.get_checkpoint_name('web_collections_shows'))
