# coding: utf-8

import datetime
import re

from analytics.plotter_lib.utils import parse_urlencoded

DATE_FORMAT = '%Y-%m-%d'


def get_block_clicks_count(block, joiners):
    import baobab
    return len([
        x
        for joiner in joiners
        for x in joiner.get_events_by_block(block)
        if isinstance(x, baobab.common.Click)
    ])


def get_baobab_block_index(block, same_name=False):
    idx = 0
    child = block.parent.first_child
    while child is not None:
        if child.id == block.id:
            return idx
        child = child.next_sibling
        if same_name:
            if child.name == block.name:
                idx += 1
        else:
            idx += 1


def get_parent_block_by_name(block, parent_name):
    current_block = block
    while current_block and current_block.name != parent_name:
        current_block = current_block.parent
    return current_block


def get_request_ui(r):
    """Определяет платформу запроса r, в терминах коллекционного ui"""
    if r.IsA('TDesktopUIProperties'):
        return 'desktop'
    elif r.IsA('TTouchUIProperties'):
        return 'mobile'
    elif r.IsA('TMobileAppUIProperties'):
        return 'yandexApp'
    elif r.IsA('TPadUIProperties'):
        return 'tablet'
    return 'empty'


def mongo_id_to_timestamp(id_str):
    return int(id_str[0:8], 16)


def mongo_id_to_datestr(id_str, date_format=DATE_FORMAT):
    id_time = mongo_id_to_timestamp(id_str)
    return datetime.datetime.fromtimestamp(id_time).strftime(date_format)


USEFUL_EVENTS = {
    'finish.authors.link.submit',
    'finish.board.create',
    'finish.board.edit',
    'finish.board.subscribe',
    'finish.board.unlock',
    'finish.card.comment',
    'finish.comment.reaction',
    'finish.comment.reply',
    'finish.card.create',
    'finish.card.like',
    'finish.card.move',
    'finish.card.share',
    'finish.card.update',
    'finish.card.wish',
    'finish.channel.subscribe',
    'finish.invitation.accept',
    'finish.onboard',
    'finish.photo.daily.subscribe',
    'finish.plugin.inline',
    'finish.user.auth',
    'finish.user.edit',
    'finish.user.subscribe',
    'click.social.share'
}


def is_useful_event(event):
    return event.get('path') in USEFUL_EVENTS


BASE_YANDEX_DOMAIN_REGEXP = r'https?://yandex\.(?:ru|by|ua|kz|com|com\.tr|com\.am|com\.ge|az|co\.il|kg|lv|lt|md|tj|tm|uz|ee|fr)'


def get_card_id_from_url(url):
    matches = re.search(BASE_YANDEX_DOMAIN_REGEXP + r'/collections/card/([^/^?]+)', url)
    if matches:
        return parse_urlencoded(matches.group(1))


def get_user_login_board_slug_from_url(url):
    matches = re.search(BASE_YANDEX_DOMAIN_REGEXP + r'/collections/user/([^/]+)/?([^/^?]+)?', url)
    if matches:
        return matches.group(1), matches.group(2)
    else:
        return None, None


def get_collections_page(url):
    """Определяет страницу коллекций по урлу"""
    # TODO: тесты
    if not isinstance(url, basestring):
        return None

    if get_card_id_from_url(url):
        return 'card'

    user_login, board_slug = get_user_login_board_slug_from_url(url)
    if user_login and board_slug:
        return 'board'
    elif user_login:
        return 'profile'
    elif re.match(BASE_YANDEX_DOMAIN_REGEXP + r'/collections/user/$', url):
        return 'myProfile'
    elif re.match(BASE_YANDEX_DOMAIN_REGEXP + r'/collections($|/$|\?|/\?)', url):
        return 'main'
    elif re.match(BASE_YANDEX_DOMAIN_REGEXP + r'/collections/feed($|/$|\?|/\?)', url):
        return 'feed'
    elif re.match(BASE_YANDEX_DOMAIN_REGEXP + r'/collections/bro/', url):
        return 'broFeed'
    elif re.match(BASE_YANDEX_DOMAIN_REGEXP + r'/images($|/$|\?|/\?)', url):
        return 'imagesFeed'
    elif re.match(BASE_YANDEX_DOMAIN_REGEXP + r'/images/touch($|/$|\?|/\?)', url):
        return 'imagesTouchFeed'
    elif re.match(BASE_YANDEX_DOMAIN_REGEXP + r'/collections/search/boards($|/$|\?|/\?)', url):
        return 'searchBoards'
    elif re.match(BASE_YANDEX_DOMAIN_REGEXP + r'/collections/channel/.+', url):
        return 'channel'
    elif re.match(BASE_YANDEX_DOMAIN_REGEXP + r'/collections/contest/photo-of-the-day/', url):
        return 'dailyPhoto'
    elif re.match(BASE_YANDEX_DOMAIN_REGEXP + r'/collections/contest/.+', url):
        return 'contest'
    return None


def format_shows_location(service='', ui='', page_name='', location='', content_type='', content_subtype=''):
    return '{}_{}_{}_{}_{}_{}'.format(service, ui, page_name, location, content_type, content_subtype)


def format_client_name(client_name):
    if not client_name:
        return 'empty'
    if 'searchplugin' in client_name or 'ru.yandex.mobile' in client_name or client_name == 'yandexsearch':
        return 'yandexApp'
    return client_name


def format_client_ui(client_ui):
    if not client_ui:
        return 'empty'
    if 'desktop' in client_ui:
        return 'desktop'
    if 'touch' in client_ui:
        return 'mobile'
    if 'pad' in client_ui:
        return 'tablet'
    return client_ui


def format_card_origin(card_origin_client_name, card_origin_action, card_origin_ui):
    origin_client_name = format_client_name(card_origin_client_name)
    origin_action = card_origin_action if card_origin_action else 'empty'
    origin_ui = format_client_ui(card_origin_ui)

    return '{}_{}_{}'.format(origin_client_name, origin_action, origin_ui)


def readable_shows_location(service='', page_name='', location='', content_type='', content_subtype=''):
    if service == 'web':
        if location == 'kokos':
            return 'серп: кокос'
        if location == 'collections-snippet':
            return 'серп: спецсниппет'
        if location == 'serpviewer':
            return 'серп: просмотрщик'
        if location == 'organic':
            return 'серп: органика {}'.format(content_type)

    if service == 'images':
        if location == 'kokoko':
            return 'картинки: кококо'

    if service == 'morda':
        if location == 'informer':
            if content_type == 'card':
                return 'морда: информер карточки'
            else:
                return 'морда: информер'

    if service == 'collections':
        if location == 'page':
            return 'коллекции: страница {}'.format(page_name)

        if location == 'similarCardToBoard':
            return 'коллекции: card2board'

        if location == 'similarBoardToBoard' or content_subtype == 'footerBoardSimilar':
            return 'коллекции: board2board'

        if location == 'channel' and content_subtype == 'channelCards':
            return 'коллекции: карточки в каналах'

        if location == 'cardPopup':
            return 'коллекции: cardPopup'

        if content_subtype == 'feedCards':
            return 'коллекции: card2user'

        if content_subtype == 'feedBoards':
            return 'коллекции: board2user'

        if content_subtype == 'boardBoards':
            return 'коллекции: борды в профиле'

        if content_subtype == 'feedRecommendations':
            return 'коллекции: рекомендации board2user в ленте card2user'

        if content_subtype == 'contestCards':
            return 'коллекции: карточки конкурсов'

        if content_subtype == 'searchCards':
            return 'коллекции: searchCards'

        if content_subtype == 'mainCards':
            return 'коллекции: mainCards'

        if content_subtype == 'boardCards':
            return 'коллекции: карточки внутри борды'

        if content_subtype == 'cards$cardSimilarToCard' or content_subtype == 'cards$similar' or content_subtype == 'cardSimilar' or content_subtype == 'cardSimilarToCard':
            return 'коллекции: card2card'

    return '{}_{}_{}_{}_{}'.format(service, page_name, location, content_type, content_subtype)
