#!/usr/bin/env python
# coding: utf-8

import io
import sys
import json
import time
import requests
import argparse
from datetime import datetime, timedelta
from shutil import copyfile

import nirvana.job_context as nv


def create_operation(pool_id, token):
    def create_operation_impl(pool_id, token):
        try:
            return requests.post(
                'https://toloka.yandex.com/api/staging/analytics-2/',
                verify=False,
                headers={
                    'Authorization': 'OAuth ' + token,
                    'Content-type': 'application/json',
                    'Accept': 'application/json',
                },
                data=json.dumps([{
                    "subject": "POOL",
                    "subject_id": pool_id,
                    "name": "approved_assignments_count",
                }])
            ).json()['id']
        except:
            pass
        return None

    TRY_COUNT = 20
    current_step = 0

    time.sleep(10)
    while current_step < TRY_COUNT:
        operation_id = create_operation_impl(pool_id, token)
        if operation_id:
            return operation_id
        time.sleep(10)
        current_step += 1
    return None


def get_approved_assignments_count(operation_id, token):
    def get_operation_info(operation_id, token):
        try:
            r = requests.get(
                'https://toloka.yandex.com/api/v1/operations/' + operation_id,
                verify=False,
                headers={'Authorization': 'OAuth ' + token}
            )
            return r.json()
        except:
            pass
        return {'status': 'ERROR'}

    TRY_COUNT = 10
    current_step = 0

    while current_step < TRY_COUNT:
        info = get_operation_info(operation_id, token)
        if info['status'] == 'SUCCESS':
            return info['details']['value'][0]['result']
        else:
            time.sleep(10)
            current_step += 1
    return 0


def toloka_wait_tasksuites(pool_id, tasksuites_count, token, max_wait_hours=24, refresh_interval_minutes=5):
    start_date = datetime.now()
    current_date = datetime.now()

    ctx = nv.context()

    print('{}, START'.format(current_date))
    while current_date - start_date < timedelta(hours=max_wait_hours):
        current_date = datetime.now()
        time.sleep(refresh_interval_minutes)

        operation_id = create_operation(pool_id, token)
        if not operation_id:
            continue
        time.sleep(10)
        approved_assignments_count = get_approved_assignments_count(operation_id, token)
        if approved_assignments_count >= tasksuites_count:
            print('OK!\n{}, tasksuites: {}'.format(current_date, approved_assignments_count))
            break
        else:
            print('{}, tasksuites: {}'.format(current_date, approved_assignments_count))

        if approved_assignments_count > 0:
            with io.open(ctx.get_status().get_log(), mode='wt', encoding='utf-8') as f:
                f.write(u'=== IN PROGRESS, DONE {}% {}/{} ===\n'.format(
                    int(round(100.0 * approved_assignments_count / tasksuites_count)),
                    approved_assignments_count,
                    tasksuites_count,
                ))
    return approved_assignments_count


def main(argv):
    parser = argparse.ArgumentParser()
    parser.add_argument('--tasksuites_count', type=int)
    parser.add_argument('--token')
    parser.add_argument('--fail_on_notenough', action='store_true')
    parser.add_argument('--max_wait_hours', default=24, type=float)
    parser.add_argument('--refresh_interval_minutes', default=5, type=float)
    args = parser.parse_args(argv[1:])

    ctx = nv.context()
    in_path = ctx.get_inputs().get('in')

    with open(in_path) as f:
        data = json.load(f)

    result = toloka_wait_tasksuites(
        data[0]['poolId'],
        args.tasksuites_count,
        args.token,
        args.max_wait_hours,
        args.refresh_interval_minutes
    )

    if result > 0 and result < args.tasksuites_count and args.fail_on_notenough:
        with io.open(ctx.get_status().get_error_msg(), mode='wt', encoding='utf-8') as f:
            f.write(u'=== ERROR, DONE {}% {}/{} ===\n'.format(
                int(round(100.0 * result / args.tasksuites_count)),
                result,
                args.tasksuites_count,
            ))
        sys.exit(1)

    with io.open(ctx.get_status().get_success_msg(), mode='wt', encoding='utf-8') as f:
        f.write(u'=== DONE {}% {}/{} ===\n'.format(
            int(round(100.0 * result / args.tasksuites_count)),
            result,
            args.tasksuites_count,
        ))

    out_path = ctx.get_outputs().get('out')
    copyfile(in_path, out_path)


if __name__ == '__main__':
    main(sys.argv)
