import numpy as np
import math

wizard_dict = {
    1002: 'tag',
    1003: 'clothes',
    1004: 'sites',
    1005: 'other_sizes',
    1006: 'ocr',
    8: 'market',
    89: 'obj_ans'
}

depth_dict = {
    'similar': 10,
    'tag': 3,
    'clothes': 5,
    'market': 5,
    'ocr': 1,
    'other_sizes': 3,
    'obj_ans': 1,
}

enrichment_dict = {
    'similar': ['judgements.similar_relevance'],
    'tag': ['judgements.tag_3_gradations', 'judgements.tag_4_gradations', 'judgements.tag_dup_3'],
    'clothes': ['judgements.octopoda_clothes_relevance'],
    'market': ['judgements.octopoda_market_relevance'],
    'ocr': [],
    'other_sizes': [],
    'obj_ans': ['judgements.octopoda_entity_relevance']
}


def squeeze(serp):
    intent_groups = []
    if 'components' not in serp:
        return

    current_wizard = None
    inside_intent_depth = 0
    for component in serp['components']:
        if component['componentInfo']['alignment'] != 3:
            continue

        if 'wizardType' in component['componentInfo']:
            if current_wizard == 'similar':
                break

            if wizard_dict[component['componentInfo']['wizardType']] != current_wizard:
                intent_groups.append(
                    {
                        'intent_name': wizard_dict[component['componentInfo']['wizardType']],
                        'components': []
                    }
                )
                current_wizard = wizard_dict[component['componentInfo']['wizardType']]
                inside_intent_depth = 0

        elif component['componentInfo']['type'] == 1:
            if current_wizard != 'similar':
                intent_groups.append(
                    {
                        'intent_name': 'similar',
                        'components': []
                    }
                )
                current_wizard = 'similar'
                inside_intent_depth = 0

        if current_wizard == 'similar' and inside_intent_depth >= 10:
            break

        if current_wizard and inside_intent_depth >= depth_dict[current_wizard]:
            continue

        info = {}
        for enrich in enrichment_dict[current_wizard]:
            info[enrich] = component.get(enrich)
        intent_groups[-1]['components'].append(info)

        inside_intent_depth += 1

    toloka_rank = sorted(
        [
            {
                'name': key, 'value': value
            } for key, value in json.loads(serp['serp_query_params.intent_weights'][0]).iteritems()
        ],
        key=lambda x: -x['value']
    )

    return {
        'query': serp['query']['text'],
        'intent_groups': intent_groups,
        'toloka_rank': toloka_rank,
        'has_annotation': int(serp.get('serp_query_param.annotation') is not None)
    }



def squeeze_serpset(serpset):
    result = []
    for serp in serpset:
        squeezed = squeeze(serp)
        result.append(squeeze(serp))

    return result

def get_similar_score(comps):

    similar_scores =  {
        'DUPLICATE': 1,
        'REL_PLUS': 1,
        'REL_MINUS': 0.3,
        'DIFFERENT': -0.5,
        '_404': 0,
    }

    depth = 10
    result = 0.0
    max_dcg = 0.0

    for i in range(depth):
        if i < len(comps) and comps[i].get('judgements.similar_relevance'):
            result += similar_scores[comps[i]['judgements.similar_relevance']['name']] / float(math.log(i + 2, 2))
            if comps[i]['judgements.similar_relevance'] == 'DUPLICATE':
                similar_scores['DUPLICATE'] /= 2.0
        max_dcg += 1 / float(math.log(i + 2, 2))

    return result / float(max_dcg)

def get_tag_score(comps, has_annotation, dup_penalty):
    tag_3_scores = {
        'USEFUL': 0.5,
        'GENERAL': 0.3,
        'STUPID': -0.5
    }

    tag_4_scores = {
        'USEFUL': 1.0,
        'RELEVANT_PLUS': 0.3,
        'RELEVANT_MINUS': -0.3,
        'IRRELEVANT': -0.5
    }

    depth = 3
    result = 0.0
    max_dcg = 0.0

    for i in range(depth):
        if i < len(comps):
            enrich_name = 'judgements.tag_3_gradations'
            rel_scores = tag_3_scores
            if has_annotation:
                enrich_name = 'judgements.tag_4_gradations'
                rel_scores = tag_4_scores

            if comps[i].get(enrich_name):
                penalty = 1.0
                if comps[i].get('judgements.tag_dup_3') and comps[i]['judgements.tag_dup_3'].get('name') == 'DUP':
                    penalty *= dup_penalty

                result += rel_scores[comps[i][enrich_name]['name']] * penalty / float(math.log(i + 2, 2))

        max_dcg += 1 / float(math.log(i + 2, 2))

    return result / float(max_dcg)


def get_clothes_score(comps):

    clothes_scores =  {
        'SAME': 1,
        'SIMILAR': 1,
        'RELATED': 0.3,
        '_404': 0,
        '404_': 0,
        'DIFFERENT': -0.5
    }

    depth = 5
    result = 0.0
    max_dcg = 0.0

    for i in range(depth):
        if i < len(comps) and comps[i].get('judgements.octopoda_clothes_relevance'):
            result += clothes_scores[comps[i]['judgements.octopoda_clothes_relevance']['name']] / float(math.log(i + 2, 2))

        max_dcg += 1 / float(math.log(i + 2, 2))

    return result / float(max_dcg)

def get_market_score(comps):

    market_scores =  {
        'SAME': 1,
        'RELATED_SAME_BRAND': 0.7,
        'RELATED': 0.5,
        'RELATED_DIFF_BRAND': 0.3,
        '_404': 0,
        '404_': 0,
        'DIFFERENT': -1.0
    }

    depth = 5
    result = 0.0
    max_dcg = 0.0

    for i in range(depth):
        if i < len(comps) and comps[i].get('judgements.octopoda_market_relevance'):
            result += market_scores[comps[i]['judgements.octopoda_market_relevance']['name']] / float(math.log(i + 2, 2))

        max_dcg += 1 / float(math.log(i + 2, 2))

    return result / float(max_dcg)


def get_ocr_score(comps):
    return 1.0


def get_other_sizes_score(comps):
    return 0.7

def get_obj_ans_score(comps):
    obj_scores = {
        'OK': 1,
        'BAD': -1,
        '_404': 0
    }
    if len(comps) > 0 and comps[0].get('judgements.octopoda_entity_relevance'):
        return obj_scores[comps[0]['judgements.octopoda_entity_relevance']['name']]

    return 0


scores  = {
    'similar': get_similar_score,
    'tag': get_tag_score,
    'clothes': get_clothes_score,
    'market': get_market_score,
    'ocr': get_ocr_score,
    'other_sizes': get_other_sizes_score,
    'obj_ans': get_obj_ans_score,
}

def get_component_scores(serp, dcg_type, dup_penalty, scores):
    result = 0.0
    max_dcg = 0.0
    current_denom = 1.0
    had_tag = 0

    component_scores = dict([key, 0.0] for key, _ in scores.iteritems())

    if dcg_type == 'dcg_full':
        current_denom = 0.5
    if dcg_type == 'dcg_tag':
        current_denom = 0.5

    for idx, group in enumerate(serp['intent_groups']):
        intent_weight = dict([[x['name'], x['value']] for x in serp['toloka_rank']])[group['intent_name']]

        if group['intent_name'] == 'tag':
            intent_score = scores['tag'](group['components'], serp['has_annotation'], dup_penalty)
            had_tag = 1
        else:
            intent_score = scores[group['intent_name']](group['components'])

        if intent_score > 0:
            result += intent_score * current_denom * intent_weight
            component_scores[group['intent_name']] += intent_score * current_denom * intent_weight
        else:
            result += intent_score * current_denom * (1.0 - intent_weight)
            component_scores[group['intent_name']] += intent_score * current_denom * (1.0 - intent_weight)

        if dcg_type == 'dcg_full' or (dcg_type == 'dcg_tag' and not had_tag):
            current_denom = 0.5 / float(math.log(idx + 3, 2))
        elif dcg_type == 'dcg_tag' and had_tag:
            current_denom = 0.5 / float(math.log(idx + 2, 2))
        elif dcg_type == 'hyp':
            current_denom = 1 / float(idx + 2)
        else:
            current_denom /= 2.0

    component_scores['score'] = result

    return component_scores


def recalc(serpset, props):

    squeezed = squeeze_serpset(serpset)

    results = []
    for serp in squeezed:
        if serp:
            results.append(get_component_scores(serp, 'dcg_full', 0.5, scores))

    final_scores = {}
    for key in scores.keys() + ['score']:
        final_scores[key] = np.mean([x[key] for x in results])

    serp_date = props[0]['date'].split('T')[0]
    final_scores['fielddate'] = serp_date

    return [final_scores]


def main(*args):
    in1, in2, in3, token, any_param, html_file = args
    return recalc(in1, in2)
