#!/usr/bin/env python
# -*- coding: utf-8 -*-

import json
import subprocess
import shlex

import yt.wrapper as yt


class Sender(object):
    def __init__(
            self,
            uids_yt_folder,
            yt_cluster='hahn',
            uids_file='uids.txt',
            out_file='uids.out',
            err_file='uids.err',
            comment='PTANALYTICS-126'
    ):
        yt.config["proxy"]["url"] = yt_cluster
        yt.config["auto_merge_output"]["action"] = "merge"
        self.uids_yt_folder = uids_yt_folder
        self.yt_tables = self._get_yt_tables()
        self.uids = set()
        self.uids_file = uids_file
        self.out_file = out_file
        self.err_file = err_file
        self.comment = comment

    def _get_yt_tables(self):
        yt_tables = []
        for table_name in json.loads(yt.get(self.uids_yt_folder, format=yt.JsonFormat())).iterkeys():
            table_path = yt.ypath_join(self.uids_yt_folder, table_name)
            print('Found table: {0}'.format(table_path))
            yt_tables.append(table_path)
        return yt_tables

    def _set_uids(self):
        for table in self.yt_tables:
            for record in yt.read_table(table):
                self.uids.add(record['uid'])

    def _write_uids_file(self):
        with open(self.uids_file, 'w') as outfile:
            for uid in self.uids:
                outfile.write('{0}\n'.format(uid))

    @staticmethod
    def execute_command(command, stdin=None):
        com_exec = subprocess.Popen(shlex.split(command), stdin=subprocess.PIPE,
                                    stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        if stdin:
            out, err = com_exec.communicate(input=stdin)
        else:
            out, err = com_exec.communicate()
        return out, err

    def _send(self):
        cmd = 'sudo /opt/bin/admkarma.py -a logout -c"{0}" -i {1} -o {2} -e {3} --max_change_frequency 1'.format(
            self.comment, self.uids_file, self.out_file, self.err_file
        )
        out, err = self.execute_command(cmd)
        if out:
            print('Out: {0}'.format(out))
        if err:
            print('Err: {0}'.format(err))

        print('На смену пароля отправлено {0} уидов.'.format(len(self.uids)))

    def _remove_yt_tables(self):
        for table in self.yt_tables:
            yt.remove(table, force=True)

    def run(self):
        if not self.yt_tables:
            return

        self._set_uids()
        if not self.uids:
            return

        if len(self.uids) > 10000:
            raise Exception('Uids count {0} more than 10000.'.format(len(self.uids)))

        self._write_uids_file()
        self._send()
        self._remove_yt_tables()


if __name__ == '__main__':
    yt_folders = ['//home/ecosystem/_squeezes_/brute_detector/af_transactions/output',
                  '//home/ecosystem/_squeezes_/brute_detector/blackbox_events/output',
                  '//home/ecosystem/_squeezes_/brute_detector/yadex_taxi_checker/output',
                  '//home/ecosystem/_squeezes_/brute_detector/emails/output_pwd_change']
    for folder in yt_folders:
        sender = Sender(folder)
        sender.run()
