#!/usr/bin/env python
# -*- coding: utf-8 -*-

import json
import logging
import subprocess
import shlex
import traceback
from Queue import Queue
from threading import Thread

import requests
import yt.wrapper as yt

requests.packages.urllib3.disable_warnings()


def execute_command(command, stdin=None):
    com_exec = subprocess.Popen(shlex.split(command), stdin=subprocess.PIPE,
                                stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if stdin:
        out, err = com_exec.communicate(input=stdin)
    else:
        out, err = com_exec.communicate()
    return out, err


def raw_login_from_email(email):
    return email.split('@', 1)[0] if '@' in email else email


def domain_from_email(email):
    if '@' not in email:
        return ''
    domain = email.split('@', 1)[1]
    return domain


def is_email_mask_matched(email, email_mask):
    minimal_masked_length = 5
    maximum_showed_symbols = 5
    email_domain = domain_from_email(email)
    mask_domain = domain_from_email(email_mask)

    if (not email_domain) or (not mask_domain) or (email_domain != mask_domain):
        return False

    email_login = raw_login_from_email(email)
    mask_login = raw_login_from_email(email_mask)

    if len(email_login) != len(mask_login):
        return False

    if (len(email_login) <= minimal_masked_length) and (len(mask_login) <= minimal_masked_length):
        return True

    login_transformed = email_login[:-minimal_masked_length]
    symbols_to_show = min(len(login_transformed), maximum_showed_symbols)
    login_transformed = login_transformed[:symbols_to_show] + '*' * (minimal_masked_length + len(login_transformed)
                                                                     - symbols_to_show)

    if login_transformed == mask_login:
        return True

    return False


class EmailsGetter(object):
    def __init__(self, input_yt_folder, threads_num=20):
        yt.config["proxy"]["url"] = "hahn"
        yt.config["auto_merge_output"]["action"] = "merge"
        self.input_yt_folder = input_yt_folder
        self.yt_tables = self._get_yt_tables()
        self.threads_num = threads_num
        self.input_queue = self._get_input_queue()
        self.output_queue = Queue()
        self.url_template = 'http://blackbox.yandex.net/blackbox/?method=userinfo&&userip=127.0.0.1&uid={0}&' \
                            'getemails=all&email_attributes=1,6&format=json&captcha=no'
        serv_ticket, _ = execute_command(
            '/usr/bin/tvmknife get_service_ticket client_credentials -s *** -d *** -S *** -q')
        self.headers = {'X-Ya-Service-Ticket': serv_ticket.rstrip()}

    def _get_yt_tables(self):
        yt_tables = []
        for table_name in json.loads(yt.get(self.input_yt_folder, format=yt.JsonFormat())).iterkeys():
            table_path = yt.ypath_join(self.input_yt_folder, table_name)
            print('Found table: {0}'.format(table_path))
            yt_tables.append(table_path)
        return yt_tables

    def _get_input_queue(self):
        input_queue = Queue()
        for table in self.yt_tables:
            for record in yt.read_table(table):
                input_queue.put({'uid': str(record['uid']), 'email': record['address']})
        return input_queue

    def _run_multi_threading(self, func, queue):
        for _ in xrange(self.threads_num):
            worker = Thread(target=func)
            worker.setDaemon(True)
            worker.start()
        queue.join()

    def _worker(self):
        while True:
            data = self.input_queue.get()
            if not data:
                self.input_queue.task_done()
                return
            uid, email = data['uid'], data['email']
            blackbox_url = self.url_template.format(uid)
            try:
                blackbox_data = requests.get(blackbox_url, headers=self.headers, timeout=10).json()
                emails = blackbox_data['users'][0]['emails']
                for item in emails:
                    full_email = item['attributes']['1']
                    if is_email_mask_matched(full_email, email):
                        is_unsafe = bool(item['attributes'].get('6'))
                        self.output_queue.put({
                            'uid': uid,
                            'email': email,
                            'full_email': full_email,
                            'is_unsafe': is_unsafe
                        })
                self.input_queue.task_done()
            except Exception:
                self.input_queue.task_done()
                continue

    def run(self):
        self._run_multi_threading(func=self._worker, queue=self.input_queue)

    def get_full_emails(self):
        return list(self.output_queue.queue)

    def get_yt_tables(self):
        return self.yt_tables


class ResetEmails(object):
    def __init__(self, full_emails, threads_num=3, log_file=None):
        self.threads_num = threads_num
        self.full_emails = full_emails
        self.input_queue = self._get_input_queue()
        self.output_queue = Queue()
        self.serv_ticket, _ = execute_command(
            '/usr/bin/tvmknife get_service_ticket client_credentials -s *** -d *** -S *** -q')
        self.url = 'https://passport-internal.yandex.ru/1/bundle/account/reset/email/?consumer=passport_antifraud_team'
        self.user_agent = 'ResetEmailClient'
        self.headers = {'X-Ya-Service-Ticket': self.serv_ticket.rstrip(), 'User-Agent': self.user_agent}
        self.log_file = log_file
        self._setup_logger()

    def _setup_logger(self):
        if self.log_file is not None:
            logging.basicConfig(
                format='%(levelname)-4s [%(asctime)s] %(message)s',
                level=logging.INFO,
                filename=self.log_file
            )
            self.logger = logging.getLogger(__name__)

    def _get_input_queue(self):
        input_queue = Queue()
        for item in self.full_emails:
            input_queue.put({'uid': str(item['uid']), 'email': item['full_email']})
        return input_queue

    def _run_multi_threading(self, func, queue):
        for _ in xrange(self.threads_num):
            worker = Thread(target=func)
            worker.setDaemon(True)
            worker.start()
        queue.join()

    def unbind(self, uid, email):
        data = {'uid': uid, 'email': email}
        req = requests.post(self.url, data=data, headers=self.headers, timeout=100, verify=False)
        self.logger.info('data: {0}, api response: {1}, response code {2}:'.format(
            json.dumps(data), json.dumps(req.json()), req.status_code))

    def _worker(self):
        while True:
            data = self.input_queue.get()
            if not data:
                self.input_queue.task_done()
                return
            uid, email = data['uid'], data['email']
            try:
                self.unbind(uid, email)
                self.input_queue.task_done()
            except Exception:
                self.logger.error('Error during processing {0}, traceback: {1}'.format(
                    json.dumps(data), traceback.print_exc()))
                self.input_queue.task_done()
                continue

    def run(self):
        self._run_multi_threading(func=self._worker, queue=self.input_queue)


if __name__ == '__main__':
    yt_folders = ['//home/ecosystem/_squeezes_/brute_detector/emails/output_email_remove']
    for folder in yt_folders:
        emails_getter = EmailsGetter(folder)
        emails_getter.run()
        full_emails = emails_getter.get_full_emails()
        if not full_emails:
            print('Full emails list for folder {0} is empty!'.format(folder))
            continue

        reset_emails = ResetEmails(full_emails=full_emails, log_file='auto.log')
        reset_emails.run()

        for table in emails_getter.get_yt_tables():
            yt.remove(table, force=True)
