from nile.api.v1 import (
    cli,
    Record,
    aggregators as na,
    filters as nf,
    extractors as ne,
    clusters,
    with_hints,
    multischema
)


from nile.api.v1 import datetime as ndt

from qb2.api.v1.typing import *

@cli.statinfra_job
def make_job(job, options, nirvana):
    input_table = nirvana.input_tables[0]
    inp = job.table(input_table)
    date = options.dates[0]

    ZERO_TIME = '0001-01-01T00:00:00Z'

    def seconds_diff(a, b):
        try:
            adt = ndt.Datetime.from_iso(a.split('+')[0][:26])
            bdt = ndt.Datetime.from_iso(b.split('+')[0][:26])
            return (bdt-adt).total_seconds()
        except:
            return None


    def get_field(rec, field):
        if field in rec:
            return rec[field]
        elif '_rest' in rec and field in rec._rest:
            return rec._rest[field]
        else:
            return None


    base_fields = dict()
    dimensions = [
        'platform_from', 'platform_to',
        'host_from', 'host_to'
        # 'service_from', 'service_to',
        # 'country_from', 'country_to',
        # 'chat_origin'
    ]
    for key in dimensions:
        base_fields[key] = None

    measures = [
        'invalid',
        'not_accepted',
        'expired',
        'failed',
        'ended',
        'msg_makecall',
        'msg_acceptcall',
        'msg_endcall'
    ]
    for key in measures:
        base_fields[key] = False

    timing_info = [
        'time_created',
        'time_accepted',
        'time_ice_completed',
        'time_ended'
    ]
    for key in timing_info:
        base_fields[key] = ZERO_TIME

    user_ids = [
        'caller_id',
        'callee_id'
    ]
    for key in user_ids:
        base_fields[key] = None

        # final_measures = ['initiated', 'accepted', 'invalid_ongoing',
        #   'valid_ongoing', 'invalid_ended', 'valid_ended', 'ongoing',
        #   'ended', 'invalid', 'valid', 'expired', 'failed', 'not_accepted'
        # ]


    @with_hints(
        output_schema=dict(
            call_guid=Optional[String],
            call=Optional[Yson],
            ts=Optional[String],
            msg=Optional[String],
            device_info=Optional[Yson],
            fielddate=String,
            _rest=Optional[Yson]
        )
    )
    def guid_date_mapper(recs):
        for rec in recs:
            call_guid = get_field(rec, 'call_guid')
            if not call_guid and isinstance(get_field(rec, 'call'), dict):
                call_guid = get_field(rec, 'call').get('call_guid')
            if call_guid is not None:
                yield Record(rec, call_guid=call_guid, fielddate=date)


    complex_out_schema = dict(
        call=Optional[Yson],
        call_duration=Optional[Float],
        call_guid=String,
        callee_id=Optional[String],
        caller_id=Optional[String],
        ended=Bool,
        expired=Bool,
        failed=Bool,
        invalid=Bool,
        msg_acceptcall=Bool,
        msg_endcall=Bool,
        msg_makecall=Bool,
        not_accepted=Bool,
        errors=Optional[List[String]],
        fielddate=String,
        host_from=Optional[String],
        host_to=Optional[String],
        ice_duration=Optional[Float],
        platform_from=Optional[String],
        platform_to=Optional[String],
        time_accepted=String,
        time_ended=String,
        time_created=String,
        time_ice_completed=String
    )

    error_out_schema = dict(
        call=Optional[Yson],
        call_duration=Optional[Float],
        call_guid=String,
        callee_id=Optional[String],
        caller_id=Optional[String],
        ended=Bool,
        expired=Bool,
        failed=Bool,
        invalid=Bool,
        msg_acceptcall=Bool,
        msg_endcall=Bool,
        msg_makecall=Bool,
        not_accepted=Bool,
        error=String,
        fielddate=String,
        host_from=Optional[String],
        host_to=Optional[String],
        ice_duration=Optional[Float],
        platform_from=Optional[String],
        platform_to=Optional[String],
        time_accepted=String,
        time_ended=String,
        time_created=String,
        time_ice_completed=String
    )


    @with_hints(
        output_schema=multischema(
            complex_out_schema,
            error_out_schema
        )
    )
    def reduce_to_line(groups, complex_out, error_out):

        def msg_finished(msg):
            return msg in ['End call', 'master: Finish expired call'] or 'hangup' in msg.lower() or 'decline' in msg.lower()

        def msg_error(msg):
            return 'error' in msg.lower()

        for key_, recs in groups:

            fields = dict(base_fields)

            maxts = ZERO_TIME
            curts = ZERO_TIME
            curcall = None
            errors = []

            for rec in recs:
                maxts = max(maxts, get_field(rec, 'ts'))

                if get_field(rec, 'msg') == 'switch: MakeCall request':
                    fields['msg_makecall'] = True

                    if isinstance(get_field(rec, 'device_info'), dict):
                        fields['platform_from'] = get_field(rec, 'device_info').get('platform')
                        fields['host_from'] = get_field(rec, 'device_info').get('app_name')
                    else:
                        fields['invalid'] = True
                        errors.append('No device_info on MakeCall')

                if get_field(rec, 'msg') == 'switch: AcceptCall request':
                    fields['msg_acceptcall'] = True
                    if isinstance(get_field(rec, 'device_info'), dict):
                        fields['platform_to'] = get_field(rec, 'device_info').get('platform')
                        fields['host_to'] = get_field(rec, 'device_info').get('app_name')
                    else:
                        fields['invalid'] = True
                        errors.append('No device_info on AcceptCall')

                if msg_finished(get_field(rec, 'msg')):
                    fields['msg_endcall'] = True

                if get_field(rec, 'msg') == 'master: Delete finished call':
                    fields['ended'] = True

                if (
                    msg_finished(get_field(rec, 'msg')) and
                    get_field(rec, 'call_status') is not None
                ):
                    fields['failed'] = True

                if get_field(rec, 'msg') == 'master: Finish expired call':
                    fields['expired'] = True

                if get_field(rec, 'call_state') == 'EXPIRED':
                    fields['expired'] = True

                if isinstance(get_field(rec, 'call'), dict) and get_field(rec, 'ts') > curts:
                    curcall = get_field(rec, 'call')
                    curts = get_field(rec, 'ts')

            if not fields['ended'] and maxts[11:] < "23:55:00":
                errors.append('No "delete finished call" message')
                fields['ended'] = True
                fields['invalid'] = True
    #                 invalidate(fields)
                complex_out(Record(key_, errors=errors, call=curcall, **fields))
                for e in errors:
                    error_out(Record(key_, error=e, **fields))
                continue

            if not isinstance(curcall, dict):
                errors.append('Invalid call value')
                fields['invalid'] = True
    #                 invalidate(fields)
                complex_out(Record(key_, errors=errors, call=curcall, **fields))
                for e in errors:
                    error_out(Record(key_, error=e, **fields))
                continue

            tc = curcall.get('time_created')
            ta = curcall.get('time_accepted')
            tic = curcall.get('time_ice_completed')
            te = curcall.get('time_ended')
            accepted = (ta is not None and ta != ZERO_TIME)
            msg_acceptcall = ta.startswith(date)  # hangup before acceptcall
            ended = (te is not None and te != ZERO_TIME)

            caller = None
            callee = None

            if not isinstance(curcall.get('participants'), list):
                caller = curcall.get('caller')
                callee = curcall.get('callee')
            else:
                if len(curcall.get('participants')) != 2:
                    errors.append('Invalid participants list')
                else:
                    p = curcall.get('participants')
                    caller = p[0]
                    callee = p[1]
                    if (not caller.get('is_caller')):
                        caller, callee = callee, caller
                    if (not caller.get('is_caller') or callee.get('is_caller')):
                        errors.append('Invalid participants list')

            if not isinstance(caller, dict):
                errors.append("caller is not a dict")
            else:
                if not isinstance(caller.get('device_info'), dict):
                    errors.append("caller['device_info'] is not a dict")
                if (
                    (
                        not caller.get('transport_id') or
                        ((not caller['device_info'].get('platform') or
                        not caller['device_info'].get('app_name')) and
                        caller['device_info'].get('platform') != 'SIP')
                    )
                ):
                    errors.append('Insufficient info on caller')

            if not isinstance(callee, dict):
                errors.append("callee is not a dict")
            else:
                if not isinstance(callee.get('device_info'), dict):
                    errors.append("callee['device_info'] is not a dict")
                if (
                    accepted and
                    (
                        (
                            not callee.get('transport_id') or
                            not callee['device_info'].get('platform') or
                            not callee['device_info'].get('app_name')
                        )
                    )
                ):
                    errors.append('Insufficient info on callee')

            if (tc is None or tc == ZERO_TIME):
                errors.append('time_created from call is invalid')
            # if (fields['msg_acceptcall'] and not accepted): errors.append('Accept status mismatch')
            if (fields['msg_endcall'] and not ended and not fields['expired']):
                errors.append('End status mismatch')

            if ((fields['msg_endcall'] or fields['expired']) and curcall.get('call_state') not in ['ENDED', 'EXPIRED']):
                errors.append('End status mismatch')

            if curcall.get('mode') == 'ECHO_TEST':
                # DO NOTHING
                continue

            if errors:
                fields['invalid'] = True
    #                 invalidate(fields)
                complex_out(Record(key_, errors=errors, call=curcall, **fields))
                for e in errors:
                    error_out(Record(key_, error=e, **fields))
                continue

            fields['caller_id'] = caller['transport_id']
            fields['platform_from'] = caller['device_info']['platform']
            if fields['platform_from'] == 'SIP':
                fields['host_from'] = 'SIP'
            else:
                fields['host_from'] = caller['device_info']['app_name']

            fields['callee_id'] = callee.get('transport_id')
            fields['platform_to'] = callee['device_info'].get('platform')
            fields['host_to'] = callee['device_info'].get('app_name')

            if curcall.get('call_state') == 'EXPIRED':
                fields['expired'] = True
            fields['not_accepted'] = not accepted

            fields['time_created'] = tc
            fields['time_accepted'] = ta
            fields['time_ice_completed'] = tic
            fields['time_ended'] = te

            fields['call_duration'] = seconds_diff(fields['time_accepted'], fields['time_ended'])
            fields['ice_duration'] = seconds_diff(fields['time_accepted'], fields['time_ice_completed'])

    #             if not fields['ended']:
    #                 ongoing(fields)

            complex_out(Record(key_, call=curcall, **fields))




    complex_out, error_out = inp.map(guid_date_mapper).groupby('call_guid', 'fielddate').reduce(reduce_to_line)

    complex_out.put('//home/mssngr/squeeze/calls/data/Call_info_oneliner_$date')
    error_out.put('//home/mssngr/squeeze/calls/data/Call_info_error_$date')

    return job


if __name__ == '__main__':
    cli.run()
