#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import argparse
import datetime as dt
import logging
import os

import yt.wrapper as yt


def parse_args():
    parser = argparse.ArgumentParser(description="Update yt links")
    parser.add_argument(
        "--target-path",
        help="Path to YT folder with target tables",
        required=True,
    )
    parser.add_argument(
        "--link-path",
        help="Path to YT folder for links",
        required=True,
    )
    parser.add_argument(
        "--target-table-name",
        help="Suffix of target tables",
        required=True,
    )
    parser.add_argument(
        "--date-from",
        help="Start table name (yyyy-mm-dd))",
        required=True,
        type=lambda s: dt.datetime.strptime(s, "%Y-%m-%d").date(),
    )
    parser.add_argument(
        "--date-to",
        help="End table name (yyyy-mm-dd)",
        required=True,
        type=lambda s: dt.datetime.strptime(s, '%Y-%m-%d').date(),
    )
    return parser.parse_args()


def main():
    logging.basicConfig(format='[%(levelname)s:%(process)d] %(asctime)s - %(message)s', level=logging.INFO)

    cli_args = parse_args()

    day = cli_args.date_from
    while day <= cli_args.date_to:
        logging.info("Process day: %s", day)

        target_path = os.path.join(cli_args.target_path, str(day), cli_args.target_table_name)
        assert yt.exists(target_path), "Table {} doesn't exist".format(target_path)
        link_path = os.path.join(cli_args.link_path, str(day))

        logging.info("Create link %s to table %s", link_path, target_path)
        yt.link(target_path, link_path, force=True)
        logging.info("%s done", day)

        day = day + dt.timedelta(days=1)

    logging.info("Success!!!")


if __name__ == "__main__":
    main()
