from datetime import datetime
import requests
import argparse
import yt.wrapper as yt


PERFORMED_QUEUES = ['MDM', 'OEBSSUPPORT']

REQUEST_URL_TEMPLATE = 'https://hd.yandex-team.ru/feedback/admin/backend-admin.php?page=csv&method=download_csv&role={}&queue={}&datepicker_from={}&datepicker_to={}'

START_DAY = "01-01-2020" 
END_DAY = datetime.now().strftime("%d-%m-%Y")

OUTPUT_SCHEMA = [
    {"name": "ticket_id", "type": "string"},
    {"name": "author", "type": "string"},
    {"name": "date", "type": "string"},
    {"name": "rating", "type": "int64"},
    {"name": "criteria", "type": "any"},
    {"name": "comment", "type": "string"},
]

DEFAULT_TABLE_COLUMNS_COUNT = len(OUTPUT_SCHEMA)
DEFAULT_REQUEST_SUCCESS_CODE = 200


def create_requested_url(queue, start_date, end_date):
    return REQUEST_URL_TEMPLATE.format(queue, queue, start_date, end_date)

def __parse_string_to_list(item):
    if len(item) == 0:
        return []
    return item[1:-2].replace('\"', '').split(',')


def load_data(requested_url, auth_token):
    response = requests.get(requested_url, headers={'Authorization': 'OAuth {}'.format(auth_token)})
    result = None
    if response.status_code == DEFAULT_REQUEST_SUCCESS_CODE:
        result = []
        try:
            lines = response.text.split('\r\n')
            # extract header
            lines.pop(0)
            for line in lines[1:]:
                row = line.split(';')      
                if len(row) != DEFAULT_TABLE_COLUMNS_COUNT:
                    continue
                else:
                    data = {}
                    data['ticket_id'] = row[0]
                    data['author'] = row[1]
                    data['date'] = row[2]
                    data['rating'] = int(row[3])
                    data['criteria'] = __parse_string_to_list(row[4])
                    data['comment'] = row[5]
                    result.append(data)
        except ValueError:
            pass
    return result


def setup_parser(parser):
        parser.add_argument('--yt_token', required=True, help='Hahn Yt token')
        parser.add_argument('--auth_token', required=True, help='Bot token to perform requests to hd.yandex-team.ru API')
        parser.add_argument('--yt_folder', required=True, help='Directory on yt.hahn whereby tables gonna be saved')
        return parser

        
def setup_yt(args):
        yt.config["proxy"]["url"] = "hahn"
        yt.config["token"] = args.yt_token
    

def main():
    parser = argparse.ArgumentParser()
    parser = setup_parser(parser)

    args = parser.parse_args()
    #args = argparse.Namespace(
    #    yt_token='***',
    #    auth_token='***',
    #    yt_folder='***'
    #)
    setup_yt(args)

    for queue in PERFORMED_QUEUES:
        requested_url = create_requested_url(queue, START_DAY, END_DAY)
        table = load_data(requested_url, args.auth_token)
        
        save_path = args.yt_folder + '/' + queue
        yt.create("table", save_path, attributes={"schema": OUTPUT_SCHEMA}, force=True)
        yt.write_table(save_path, table, format=yt.JsonFormat(attributes={"encode_utf8": False}), raw=False,
                   force_create=True)
    

if __name__ == '__main__':
    main()
