#!/usr/bin/env python
# -*- coding: utf-8 -*-

import logging
import json
import subprocess
import shlex
import time
import traceback
from Queue import Queue
from threading import Thread

import requests
import yt.wrapper as yt

requests.packages.urllib3.disable_warnings()


def execute_command(command, stdin=None):
    com_exec = subprocess.Popen(shlex.split(command), stdin=subprocess.PIPE,
                                stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    if stdin:
        out, err = com_exec.communicate(input=stdin)
    else:
        out, err = com_exec.communicate()
    return out, err


def get_phone_mask(raw_number):
    unmasked_count = 6
    if not raw_number or len(raw_number) < unmasked_count:
        return raw_number
    return "%s%s" % (raw_number[:unmasked_count], '*' * (len(raw_number) - unmasked_count))


def is_phone_mask_matched(phone_mask, raw_number):
    return phone_mask == get_phone_mask(raw_number)


class CorpUidsGetter(object):
    def __init__(self, uids_yt_folder, threads_num=20, log_file=None):
        yt.config["proxy"]["url"] = "hahn"
        yt.config["auto_merge_output"]["action"] = "merge"
        self.uids_yt_folder = uids_yt_folder
        self.yt_tables = self._get_yt_tables()
        self.corp_phones_table = '//home/taxi-fraud/export/production/passport/corps_phones'
        self.fast_corp_phones_table = '//home/ecosystem/_squeezes_/taxi_corp_uids/fast_corp_phones'
        self.corp_phones = self._get_taxi_corp_phones()
        self.threads_num = threads_num
        self.input_queue = self._get_input_queue()
        self.output_queue = Queue()
        self.serv_ticket, _ = execute_command(
            '/usr/bin/tvmknife get_service_ticket client_credentials -s *** -d *** -S *** -q')
        self.url_template = 'http://blackbox.yandex.net/blackbox/?method=userinfo&getphones=all&' \
                            'userip=127.0.0.1&uid={0}&type=all&phone_attributes=102&format=json&captcha=no'
        self.headers = {'X-Ya-Service-Ticket': self.serv_ticket.rstrip()}
        self.log_file = log_file
        self._setup_logger()

    def _get_taxi_corp_phones(self):
        corp_phones = set()
        for record in yt.read_table(self.corp_phones_table):
            corp_phones.add(record['user_phone'])
        for record in yt.read_table(self.fast_corp_phones_table):
            corp_phones.add(record['phone_number'])
        return corp_phones

    def _setup_logger(self):
        if self.log_file is not None:
            logging.basicConfig(
                format='%(levelname)-4s [%(asctime)s] %(message)s',
                level=logging.INFO,
                filename=self.log_file
            )
            self.logger = logging.getLogger(__name__)

    def _get_yt_tables(self):
        yt_tables = []
        for table_name in json.loads(yt.get(self.uids_yt_folder, format=yt.JsonFormat())).iterkeys():
            table_path = yt.ypath_join(self.uids_yt_folder, table_name)
            print('Found table: {0}'.format(table_path))
            yt_tables.append(table_path)
        return yt_tables

    def _get_input_queue(self):
        input_queue = Queue()
        seen = {}
        for table in self.yt_tables:
            for record in yt.read_table(table):
                uid, number = record['uid'], record['number']
                seen_number = seen.get(uid)
                if seen_number and seen_number == number:
                    continue
                input_queue.put({'uid': uid, 'number': number})
                seen[uid] = number
        return input_queue

    def _run_multi_threading(self, func, queue):
        for _ in xrange(self.threads_num):
            worker = Thread(target=func)
            worker.setDaemon(True)
            worker.start()
        queue.join()

    def _worker(self):
        while True:
            data = self.input_queue.get()
            if not data:
                self.input_queue.task_done()
                return
            uid, number = data['uid'], data['number']
            blackbox_url = self.url_template.format(uid)
            try:
                blackbox_data = requests.get(blackbox_url, headers=self.headers, timeout=10).json()
                for user in blackbox_data['users']:
                    phones = user['phones']
                    for item in phones:
                        full_number = item['attributes']['102']
                        if is_phone_mask_matched(number, full_number) and full_number in self.corp_phones:
                            self.output_queue.put({'uid': uid, 'full_number': full_number})
                            self.logger.info('Found corp uid: {0}, number: {1}'.format(uid, full_number))
                self.input_queue.task_done()
            except Exception:
                self.input_queue.task_done()
                continue

    def get_corp_uids(self):
        if not self.yt_tables:
            return []
        self._run_multi_threading(func=self._worker, queue=self.input_queue)
        return list({item['uid'] for item in list(self.output_queue.queue)})

    def get_processed_tables(self):
        return self.yt_tables


class AntifraudListsApiClient(object):
    def __init__(self):
        self.api_url = 'https://fraud.so.yandex-team.ru/update_list'
        self.current_timestamp = int(time.time())
        self.after_two_years_timestamp = self.current_timestamp + 31536000 * 2
        self.data = {
            "channel": "auth",
            "sub_channel": "login",
            "list_name": "corp_uids_2fa_auto_list",
            "field": "uid",
            "from": self.current_timestamp * 1000,
            "to": self.after_two_years_timestamp * 1000,
            "items": [],
            "reason": "corp_uid",
            "author": "i-osipov"
        }
        self.serv_ticket, _ = execute_command(
            '/usr/bin/tvmknife get_service_ticket client_credentials -s *** -d *** -S *** -q')
        self.headers = {'X-Ya-Service-Ticket': self.serv_ticket.rstrip()}

    @staticmethod
    def get_list_chunks(seq, num):
        out = []
        for i in range(0, len(seq), num):
            out.append(seq[i:i + num])
        return out

    def push_corp_uids(self, uids_list, processed_tables):
        print('Corp uids count: {0}'.format(len(uids_list)))
        uids_list_chunks = self.get_list_chunks(uids_list, 9999)
        err_cnt = 0
        for chunk in uids_list_chunks:
            self.data["items"] = chunk
            try:
                requests.post(self.api_url, data=json.dumps(self.data), headers=self.headers, timeout=100, verify=False)
            except Exception:
                err_cnt += 1
                print('Error during processing {0}, traceback: {1}'.format(chunk, traceback.print_exc()))

        if err_cnt == len(uids_list):
            print('The lists were not processed!')
            return

        for table in processed_tables:
            yt.remove(table, force=True)


if __name__ == '__main__':
    corp_uids_getter = CorpUidsGetter(
        uids_yt_folder='//home/ecosystem/_squeezes_/taxi_corp_uids/output',
        log_file='taxi_corp_phones.log'
    )

    # Уиды с привязкой корпоративного номера passport-log
    taxi_corp_uids = corp_uids_getter.get_corp_uids()
    if taxi_corp_uids:
        processed_yt_tables = corp_uids_getter.get_processed_tables()
        antifraud_api_client = AntifraudListsApiClient()
        antifraud_api_client.push_corp_uids(taxi_corp_uids, processed_yt_tables)

    # Полный список корпоративных уидов
    full_list_node = '//home/ecosystem/_squeezes_/taxi_corp_uids/output_full_list'
    full_list_tables = [
        yt.ypath_join(full_list_node, table_name) for
        table_name in json.loads(yt.get(full_list_node, format=yt.JsonFormat())).iterkeys()
    ]
    full_list = set()
    for t in full_list_tables:
        for rec in yt.read_table(t):
            full_list.add(rec['uid'])

    if full_list:
        print('Таблицы с полными списками: {0}'.format(json.dumps(full_list_tables)))
        print('Количество уидов в полном списке: {0}'.format(len(full_list)))
        antifraud_api_client = AntifraudListsApiClient()
        antifraud_api_client.push_corp_uids(list(full_list), full_list_tables)
