#!/usr/bin/env python
# -*- coding: utf-8 -*-

from nile.api.v1 import (
    clusters,
    filters as nf,
    aggregators as na,
    extractors as ne,
    files as nile_files,
    with_hints,
    Record
)

from qb2.api.v1 import (
    extractors as qe,
    resources as qr,
    typing as qt
)

import json
import argparse


def make_job(start_date, end_date, job_root):
    cluster = clusters.yt.Hahn()
    cluster = cluster.env(
        templates={
            'dates': '{%s..%s}' % (start_date, end_date),
            'job_root': job_root,
            'checkpoints_root': '$job_root/checkpoints'
        },
        yt_spec_defaults={
            'pool_trees': ['physical'],
            'tentative_pool_trees': ['cloud']
        },
        parallel_operations_limit=10
    )
    
    return cluster.job()


def get_country_name(geo_id):
    """Takes geo id and returns country name based on geobase"""
    geob = qr.get('Geobase')
    try:
        region = geob.region_by_id(int(geo_id))
        parents = region.path
        country = [reg.name for reg in parents if reg.type == 3]
        if country:
            country_name = country[0]
        else:
            country_name = None
        return country_name
    except:
        return None


def get_destination_permalink(input_points):
    """
    Takes 'vars' field of the 'route.request-route' event 
    and returns its permalink value if the deistination 
    point was an organisation
    """
    points = json.loads(input_points).get('points', None)
    if points is None:
        return 0
    else:
        destination_point = points.split('~')[-1]
        if 'org?oid=' in destination_point:
            destination_permalink = \
                destination_point.split('org?oid=')[1]
            return int(destination_permalink)


def get_route_type(value):
    """Returns the mode of transport parameter value"""
    if 'route_type' in value.keys():
        route_type = value['route_type']
        return route_type
    else:
        return 'NA'


def main(start_date, end_date, job_root):
    job = make_job(start_date=start_date, end_date=end_date, job_root=job_root)

    altay_pretty = job.table(
        '//home/sprav/assay/common/company_pretty_format'
    )
    rubrics = altay_pretty.project(
        'permalink',
        rubric_names=ne.custom(lambda x: x.split(';'), 'rubric_names_ru')
    )

    mobile_log = job.table('//statbox/metrika-mobile-log/@dates')
    route_shows = mobile_log.filter(
        nf.and_(
            nf.equals('APIKey', '4'),
            nf.equals('SessionType', 'SESSION_FOREGROUND'),
            nf.equals('EventName', 'routes.show-route')
        )
    ).project(
        'DeviceID',
        'SessionID',
        show_datetime='EventDateTime',
        reqid=ne.custom(lambda x: json.loads(x).get('reqid', None),
                        'EventValue'),
        auto_opened=ne.custom(lambda x: json.loads(x).get('auto_opened', None),
                              'EventValue'),
        route_type=ne.custom(lambda x: get_route_type(json.loads(x)),
                             'EventValue')
    )

    route_requests = mobile_log.filter(
        nf.and_(
            nf.equals('APIKey', '4'),
            nf.equals('SessionType', 'SESSION_FOREGROUND'),
            nf.equals('EventName', 'route.request-route')
        )
    ).project(
        'DeviceID',
        'SessionID',
        request_datetime='EventDateTime',
        reqid=ne.custom(lambda x: json.loads(x).get('reqid', None),
                        'EventValue'),
        country=ne.custom(lambda x: get_country_name(x),
                          'RegionID'),
        permalink=ne.custom(lambda x: get_destination_permalink(x),
                            'EventValue'),
        files=[nile_files.StatboxDict('Geobasev6.bin', use_latest=True)]
    ).filter(
        nf.equals('country', 'Россия')
    ).join(
        rubrics, by='permalink'
    ).join(
        route_shows, by=['DeviceID', 'SessionID', 'reqid']
    ).project(
        'request_datetime',
        'show_datetime',
        'rubric_names',
        'permalink',
        'route_type',
        'auto_opened'
    ).sort(
        'request_datetime'
    ).put(
        '$job_root/routes_to_organisations_mobile',
        schema={
            'request_datetime': qt.String,
            'show_datetime': qt.String,
            'rubric_names': qt.List[qt.String],
            'permalink': qt.Int64,
            'route_type': qt.String,
            'auto_opened': qt.String,
        }
    )

    job.run()


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '-s',
        '-start_date',
        required=True,
        help='start date - format YYYY-MM-DD'
    )
    parser.add_argument(
        '-e',
        '-end_date',
        required=False,
        help='end date - format YYYY-MM-DD'
    )
    parser.add_argument(
        '-p',
        '-path',
        type=str,
        default='//home/geo-analytics/pruzhinkina/GEOHELL-124',
        help='path to result'
    )
    args = parser.parse_args()
    if not args.e:
        args.e = args.s

    main(args.s, args.e, args.p)
