"""module defining extracting functions"""


from datetime import datetime, timedelta

from nile.api.v1 import (
    aggregators as na,
    extractors as ne,
)
from qb2.api.v1 import (
    extractors as qe,
    filters as qf
)


def concat_filtering_source(source):
    """returns filtering search_source fields from mapreqans"""
    return (source is not None) and any(s in source for s in ['_form_', '_voice_', '_text_'])


def get_dates_interval(start_date, finish_date):
    return '{{{}..{}}}'.format(start_date, finish_date)


def extract_mapreqans(job, start_date, finish_date):
    """returns mapreqans streams"""
    start_date = str(start_date.date())
    finish_date = str(finish_date.date())

    mapreqans = job.table(
        '$mapreqans/{}'.format(get_dates_interval(start_date, finish_date))
    ).qb2(
        log='map-reqans-log',
        fields=[
            'search_source',
            'origin',
            'serpid',
            'raw_uuid',
            'date',
            'search_type'
        ],
        filters=[
            qf.equals('origin_project', 'Mobile_Soft_Maps'),
            qf.compare('date', '>=', start_date),
            qf.compare('date', '<=', finish_date)
        ],
        mode='yamr_lines'
    ).filter(
        qf.custom(concat_filtering_source, 'search_source')
    ).project(
        ne.all(exclude=['search_type']),
        qe.unfold_prefixes('search_type_prefixes', 'search_type')
    ).project(
        ne.all(exclude=['search_type_prefixes']),
        search_type=ne.custom(
            lambda search_type_prefixes: '/'.join(search_type_prefixes)
        ),
    ).groupby(
        'search_type',
        'origin',
        'raw_uuid',
        'date'
    ).aggregate(
        requests=na.count(),
        serpids=na.count_distinct('serpid', in_memory=True)
    )

    return mapreqans


def extract_appmetrica(job, start_date, finish_date):
    """returns appmetrica platform and UUID for join streams"""

    start_date_stock = str((start_date - timedelta(days=1)).date())
    finish_date_stock = str((min(start_date + timedelta(days=7),
                                 datetime.now() - timedelta(days=1))
                             ).date())

    appmetrica = job.table(
        '$mobmetrika/{}'.format(
            get_dates_interval(start_date_stock, finish_date_stock)
        )
    ).qb2(
        log='metrika-mobile-log',
        fields=[
            'app_platform',
            'uuid'
        ],
        filters=[
            qf.equals('app_project', 'Mobile_Soft_Maps'),
            qf.compare('session_start_date', '>=', str(start_date.date())),
            qf.compare('session_start_date', '<=', str(finish_date.date()))
        ]
    ).unique(
        'app_platform', 'uuid'
    )

    return appmetrica
