import argparse
import logging
from datetime import datetime, timedelta

from nile.api.v1 import (
    clusters
)

from extractors import extract_mapreqans, extract_appmetrica
from joiners import join_tables


def search_by_platform(start_date, finish_date, path):
    """prepare and run job"""

    cluster = clusters.yt.Hahn().env(
        templates={
            'mapreqans':
                'statbox/map-reqans-log',
            'mobmetrika':
                'logs/metrika-mobile-log/1d',
            'task_root': path
        },
        yt_spec_defaults=dict(
            pool_trees=["physical"],
            tentative_pool_trees=["cloud"]
        )
    )

    job = cluster.job('calculate-searches-by-platform').env(
        files=['extractors.py', 'joiners.py']
    )

    mapreqans_stream = extract_mapreqans(job, start_date, finish_date)
    appmetrica_stream = extract_appmetrica(job, start_date, finish_date)

    result = join_tables(mapreqans_stream, appmetrica_stream)
    result.put(
        '$task_root/{}..{}'.format(str(start_date.date()), str(finish_date.date()))
    )

    job.run()


def valid_date(date):
    try:
        return datetime.strptime(date, "%Y-%m-%d")
    except ValueError:
        msg = "Not a valid date: '{}'.".format(date)
        raise argparse.ArgumentTypeError(msg)


def checking_dates_period(start_date, finish_date):
    yesterday = datetime.now() - timedelta(days=1)
    logger = logging.getLogger('extract')
    if start_date > finish_date:
        logger.error('start date more than finish date')
    if finish_date > yesterday:
        logger.error('finish date more than yesterday')


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '-s',
        '-start_date',
        type=valid_date,
        required=True,
        help='the start date for report - format YYYY-MM-DD'
    )
    parser.add_argument(
        '-f',
        '-finish_date',
        type=valid_date,
        required=False,
        help='the finish date for report - format YYYY-MM-DD'
    )
    parser.add_argument(
        '-p',
        '-path',
        type=str,
        default='home/geo-analytics/denislina/mama-179-calculate-searches',
        help='path to result report'
    )
    args = parser.parse_args()
    if not args.f:
        args.f = args.s

    checking_dates_period(start_date=args.s,
                          finish_date=args.f)

    search_by_platform(start_date=args.s,
                       finish_date=args.f,
                       path=args.p)
