# coding=utf-8

from collections import Counter

from nile.api.v1 import (
    Record,
    files as nf
)
from qb2.api.v1 import (
    resources as qr
)

from common import MAX_SESSION_DURATION, EVENT_METRICS, EARTH_ID, detect_by_record


def session_reducer(sessions):
    for key, records in sessions:
        u_id = key.u_id
        session_id = key.session_id
        min_offset, max_offset = -1, -1
        app_platform = None
        app_version = None
        geob = qr.get('LibGeobase6')
        last_geo_id = EARTH_ID
        has_location_access = False

        session_metrics = Counter()  # все рассчитываемые метрики состоят из событийных (EVENT_METRICS) и остальных
        for name in EVENT_METRICS.keys():
            session_metrics[name] = 0
        session_metrics['authorized'] = False
        session_metrics['map_caches'] = 0
        session_metrics['authorized_by_puid'] = False

        for record in records:
            if min_offset == -1 or min_offset > int(record.event_time_offset):
                min_offset = int(record.event_time_offset)
            if max_offset == -1 or max_offset < int(record.event_time_offset):
                max_offset = int(record.event_time_offset)
                last_geo_id = int(record.geo_id)

            if record.LocationSource == 'GPS':  # если хотя бы 1 координата в сессии определена по GPS, то доступ есть
                has_location_access = True

            for name, conditions in EVENT_METRICS.iteritems():
                if detect_by_record(conditions)(record):
                    session_metrics[name] += 1  # все событийные метрики - это подсчет удовлетворяющих предикату событий
                    break
            else:
                if record.event_name == 'application.start-session' and \
                                (record.event_value or {}).get('authorized') == 'true':
                    session_metrics['authorized'] = 1
                if record.event_name == 'application.get-global-paramethers':
                    session_metrics['map_caches'] = int((record.event_value or {}).get('map_caches') or '0')
            if record.puid:
                session_metrics['authorized_by_puid'] = 1
            app_platform = record.app_platform
            app_version = record.app_version

        region_id = EARTH_ID  # регион сессии определяем как регион последнего события в сессии, по дефолту - Земля
        if has_location_access:  # если нет доступа к локации, то регион остается Земля
            geo_id = last_geo_id if last_geo_id else EARTH_ID
            while geo_id != EARTH_ID:
                try:
                    region = geob.get_region_by_id(geo_id)
                    if region['type'] in (3, 5, 6):
                        break
                    geo_id = region['parent_id']
                except RuntimeError:
                    geo_id = EARTH_ID
            region_id = geo_id

        yield Record(
            u_id=u_id,
            session_id=session_id,
            duration=min(max_offset - min_offset, MAX_SESSION_DURATION),  # продолжительность сессии (не больше порога)
            region_id=str(region_id),
            app_version=app_version,
            app_platform=app_platform,
            **session_metrics
        )


def user_reducer(users):
    for key, records in users:
        u_id = key.u_id

        user_metrics = Counter()
        regions = set()
        app_platforms = set()
        app_versions = set()
        user_metrics['authorized'] = False
        user_metrics['map_caches'] = 0
        user_metrics['authorized_by_puid'] = False

        for record in records:
            for name in EVENT_METRICS.keys():
                user_metrics[name] += record[name]
            user_metrics['duration'] += record['duration']
            user_metrics['count_sessions'] += 1
            user_metrics['authorized'] = max(user_metrics['authorized'], record['authorized'])
            user_metrics['map_caches'] = max(user_metrics['map_caches'], record['map_caches'])
            user_metrics['authorized_by_puid'] = max(user_metrics['authorized_by_puid'], record['authorized_by_puid'])
            regions.add(record['region_id'])
            app_platforms.add(record['app_platform'])
            app_versions.add(record['app_version'])

        yield Record(
            u_id=u_id,
            regions=','.join(regions),
            app_platforms=','.join(app_platforms),
            app_versions=','.join(app_versions),
            **user_metrics
        )


def main(stream):
    return stream.groupby(
        'u_id',
        'session_id'
    ).reduce(
        session_reducer,
        files=[nf.StatboxDict('geodata5.bin')]
    ).groupby(
        'u_id'
    ).reduce(user_reducer)
