#!/usr/bin/env python
#  -*- coding: utf-8 -*-


def check_parameter(param, value):
    def f(event_value):
        if isinstance(value, list):
            return (event_value or {}).get(param) in value
        else:
            return (event_value or {}).get(param) == value
    return f


EVENT_METRICS = {
    'make_call': [{'event_name': 'place.make-call'}],
    'panoramas': [{'event_name': ['place.show-panoramas-view', 'map.show-panoramas-view']}],
    'use_ar': [{'event_name': ['ar_object.open_ar', 'ar_object.make-route', 'map.show-ar-view', 'place.show-ar-view']}],
    'review': [{'event_name': 'place.add-review.submit'}],
    'photo': [{'event_name': 'place.add-photo.submit'}],
    'rate': [{'event_name': 'place.rate-place'}],
    'kilometers': [{'event_name': 'guidance.ping'}],
    'routes': [{'event_name': 'routes.show-route',
                'event_value': check_parameter('auto_opened', 'true')},
               {'event_name': 'route.make-route'}],
    'car_routes': [{'event_name': ['routes.show-route', 'route.make-route'],
                    'event_value': check_parameter('route_type', 'car')}],
    'pedestrian_routes': [{'event_name': 'routes.show-route',
                           'event_value': check_parameter('route_type', 'pedestrian')},
                          {'event_name': 'route.make-route',
                           'event_value': check_parameter('route_type', 'pedestrian')},
                          {'event_name': 'route.make-route',
                           'event_value': check_parameter('subroute_type', 'pedestrian')}],
    'transport_routes': [{'event_name': 'routes.show-route',
                          'event_value': check_parameter('route_type', 'transport')},
                         {'event_name': 'route.make-route',
                          'event_value': check_parameter('subroute_type', 'ot')}],
    'bike_routes': [{'event_name': 'routes.show-route',
                     'event_value': check_parameter('route_type', 'bike')}],
    'all_start_navigation': [{'event_name': 'route.start-navigation'}],
    'car_start_navigation': [{'event_name': 'route.start-navigation',
                              'event_value': check_parameter('route_type', 'car')},
                             {'event_name': 'route.start-navigation',
                              'event_value': check_parameter('type', 'car')}],
    'add_bookmarks': [{'event_name': ['add-my-place.sumbit', 'map.add-bookmark.submit']}],
    'toponym_search': [{'event_name': ['search.get-search-results', 'route-points.get-search-results', 'route.get-search-results'],
                        'event_value': check_parameter('search_type', 'toponyms')}],
    'org_search': [{'event_name': ['search.get-search-results', 'route-points.get-search-results', 'route.get-search-results'],
                    'event_value': check_parameter('search_type', ['org1', 'chain', 'rubric'])}],
    'rubric_search': [{'event_name': ['search.get-search-results', 'route-points.get-search-results', 'route.get-search-results'],
                       'event_value': check_parameter('search_type', 'rubric')}],
    'cafe': [{'event_name': ['search.open-place-view', 'search.show-place-card'],
              'event_value': check_parameter('category', ['Кафе', 'Ресторан', 'Быстрое питание', 'Пиццерия', 'Столовая', 'Суши-бар'])}],
    'gas_station': [{'event_name': ['search.open-place-view', 'search.show-place-card'],
              'event_value': check_parameter('category', ['АЗС', 'АГНС', 'АГЗС', 'АГНКС', 'Станция зарядки электромобилей'])}],
    'ATM': [{'event_name': ['search.open-place-view', 'search.show-place-card'],
              'event_value': check_parameter('category', ['Банкомат', 'Платежный терминал'])}],
    'pharmacy': [{'event_name': ['search.open-place-view', 'search.show-place-card'],
              'event_value': check_parameter('category', ['Аптека', 'Ветеринарная аптека', 'Справочная аптек'])}],
    'supermarket': [{'event_name': ['search.open-place-view', 'search.show-place-card'],
              'event_value': check_parameter('category', ['Магазин продуктов', 'Супермаркет', 'Гипермаркет', 'Магазин мяса, колбас', 'Булочная, пекарня', 'Диетические и диабетические продукты', 'Кондитерский магазин', 'Магазин кулинарии', 'Магазин чая и кофе', 'Молочный магазин', 'Овощной магазин', 'Рыбный магазин'])}],
    'pub': [{'event_name': ['search.open-place-view', 'search.show-place-card'],
              'event_value': check_parameter('category', ['Бар, паб', 'Спорт-бар'])}],
    'shopping_center': [{'event_name': ['search.open-place-view', 'search.show-place-card'],
              'event_value': check_parameter('category', ['Торговый центр', 'Универмаг', 'Гипермаркет'])}],
    'hotel': [{'event_name': ['search.open-place-view', 'search.show-place-card'],
              'event_value': check_parameter('category', ['Гостиница', 'Хостел', 'Общежитие', 'Жилье посуточно'])}],
    'cinema': [{'event_name': ['search.open-place-view', 'search.show-place-card'],
              'event_value': check_parameter('category', ['Кинотеатр'])}],
    'beauty_saloon': [{'event_name': ['search.open-place-view', 'search.show-place-card'],
              'event_value': check_parameter('category', ['Солярий', 'Салон красоты', 'Ногтевая студия', 'Парикмахерская', 'СПА-салон', 'Косметология'])}],
    'car_service': [{'event_name': ['search.open-place-view', 'search.show-place-card'],
              'event_value': check_parameter('category', ['Мойка', 'Автосервис', 'Пункт техосмотра', 'Кузовной ремонт', 'Ремонт АКПП', 'Ремонт двигателей', 'Шиномонтаж', 'Ремонт мототехники'])}],
    'fitness': [{'event_name': ['search.open-place-view', 'search.show-place-card'],
              'event_value': check_parameter('category', ['Фитнес-клуб', 'Спортивный клуб', 'секция', 'Спортивный', 'тренажерный зал'])}]
}


def get_events_set(predicates):
    events_set = set()
    for predicates_list in predicates.values():
        for predicate in predicates_list:
            events = predicate.get('event_name')
            print events
            if isinstance(events, list):
                events_set.update(set(events))
            elif events:
                events_set.add(events)
    return events_set


EARTH_ID = 10000
RUSSIA_ID = 225
MAX_SESSION_DURATION = 7200  # suppose 2 hours is maximum session duration


def _check_condition(record, condition):
    """
    Функция, проверяющая выполнение внутренних условий из условия condition,
    представляющего собой словарь. Условие может быть одним значением, списком
    или функцией.

    >>> from nile.api.v1 import Record
    >>> condition = {'event_name': 'maps_www.routes_panel',
    ...              'event_type': 'show'}
    >>> record = Record(event_name='maps_www', event_type='show')
    >>> all([i for i in _check_condition(record, condition)])
    False
    >>> condition = {'event_name': 'maps_www.routes_panel',
    ...               'var_coords': lambda c: c is not None}
    >>> record = Record(event_name='maps_www.routes_panel', event_type='show')
    >>> all([i for i in _check_condition(record, condition)])
    False
    >>> condition = {'event_name': 'maps_www.routes_panel',
    ...               'list_field': ['value1', 'value2']}
    >>> record = Record(event_name='maps_www.routes_panel',
    ...                 list_field='value1')
    >>> all([i for i in _check_condition(record, condition)])
    True

    """
    for key, value in condition.iteritems():
        if callable(value):
            yield value(record.get(key, None))
        elif isinstance(value, list):
            yield record.get(key, None) in value
        else:
            yield record.get(key, None) == value


def detect_by_record(conditions):
    """
    Функция, проводящая сравнение record'а с набором условий
    из аргумента-списка. Чтобы вернуть True, должно выполниться хотя бы одно.

    Каждое условие представляет собой словарь. Каждая пара key-value - другое
    условие, обрабатывающее только одно поле из record.
    Истинность каждого проверятся функцией _check_condition.

    Все представленные парами условия должны выполняться, чтобы считать
    выполненным условие верхнего уровня.

    >>> from nile.api.v1 import Record
    >>> conditions = [{'event_name': 'maps_www.routes_panel',
    ...                'event_type': 'show'}]
    >>> record = Record(event_name='maps_www', event_type='show')
    >>> detect_by_record(conditions)(record)
    False
    >>> conditions = [{'event_name': 'maps_www.routes_panel',
    ...                'event_type': 'show'}]
    >>> record = Record(event_name='maps_www.routes_panel', event_type='click')
    >>> detect_by_record(conditions)(record)
    False
    >>> conditions = [{'event_name': 'maps_www.routes_panel',
    ...                'event_type': 'show'}]
    >>> record = Record(event_name='maps_www.routes_panel', event_type='show')
    >>> detect_by_record(conditions)(record)
    True
    >>> conditions = [{'event_name': 'maps_www.routes_panel',
    ...                'var_coords': lambda c: c is not None}]
    >>> record = Record(event_name='maps_www.routes_panel', event_type='show')
    >>> detect_by_record(conditions)(record)
    False
    >>> conditions = [{'event_name': 'maps_www.routes_panel',
    ...                'var_coords': lambda c: c is not None}]
    >>> record = Record(event_name='maps_www.routes_panel',
    ...                 var_coords='00, 00')
    >>> detect_by_record(conditions)(record)
    True
    >>> conditions = [{'event_name': 'maps_www.routes_panel',
    ...                'list_field': ['value1', 'value2']}]
    >>> record = Record(event_name='maps_www.routes_panel',
    ...                 list_field='value1')
    >>> detect_by_record(conditions)(record)
    True
    >>> conditions = [{'event_name': 'maps_www.routes_panel',
    ...                'list_field': ['value1', 'value2']}]
    >>> record = Record(event_name='maps_www.routes_panel',
    ...                 list_field='value3')
    >>> detect_by_record(conditions)(record)
    False

    """

    def result_function(record):
        for condition in conditions:
            if all(_check_condition(record, condition)):
                return True
        return False

    return result_function
