#!/usr/bin/env python
#  -*- coding: utf-8 -*-


import argparse
import datetime
import time

from nile.api.v1 import clusters

import aggregators
import extractors


def make_dates(dates=None):
    """make a proper format dates for the job"""
    if dates is None:
        today = time.localtime()
        date_from = datetime.datetime(*today[:3]) - datetime.timedelta(7)
        date_to = datetime.datetime(*today[:3]) - datetime.timedelta(1)
        dates_bounds = '{%s..%s}' % (date_from.strftime('%Y-%m-%d'), date_to.strftime('%Y-%m-%d'))
    else:
        if '..' in dates:
            date_from, date_to = dates.split('..')
            date_from = datetime.datetime.strptime(date_from, '%Y-%m-%d').date()
            date_to = datetime.datetime.strptime(date_to, '%Y-%m-%d').date()
            dates_bounds = '{%s}' % dates
        else:  # 'dates' consist of one date
            date_from = datetime.datetime.strptime(dates, '%Y-%m-%d').date()
            date_to = datetime.datetime.strptime(dates, '%Y-%m-%d').date()
            dates_bounds = '{%s..%s}' % (dates, dates)

    return dates_bounds, date_from, date_to


def main(dates, scale, dir_path):
    dates, date_from, date_to = make_dates(dates)
    cluster = clusters.yt.Hahn().env(
        templates=dict(
            job_root='home/$user_name/nile',
            dates=dates,
            mobmetrika='//statbox/metrika-mobile-log.6p'
        ),
        files=[
            'extractors.py',
            'aggregators.py',
            'common.py'
        ]
    )
    job = cluster.job()

    # preparing logs
    stream = extractors.main(job, dates)

    # aggregating metrics
    stream.call(aggregators.main)\
        .put(dir_path+'/scale_{scale}/{start}:{end}'.format(
            scale=scale,
            start=date_from,
            end=date_to)
        )

    job.run()


if __name__ == '__main__':
    parser = argparse.ArgumentParser(
        description='''
            prepare and run a nile job on discovery collections basic metrics
            will collect all historical data'''
    )
    parser.add_argument(
        '--dates',
        '-d',
        help='''
            dates for process in format first_date..last_date,
            where each date is fromatted like %Y-%m-%d''',
        type=str,
    )
    parser.add_argument(
        '-p',
        '--path',
        type=str,
        default='home/geo-analytics/itangaev/MAMA-180',
        help='path to result tables'
    )
    parser.add_argument(
        '--scale',
        dest='scale',
        help='''Number of days in one aggregated table
             ''',
        default='7'
    )
    args = parser.parse_args()
    _, start_date, end_date = make_dates(args.dates)
    delta = int(args.scale)
    while start_date + datetime.timedelta(delta-1) <= end_date:
        dates = '%s..%s' % (start_date.strftime('%Y-%m-%d'),
                            (start_date + datetime.timedelta(delta-1)).strftime('%Y-%m-%d'))
        print 'Start calculation: ' + dates
        main(dates, args.scale, args.path)
        start_date += datetime.timedelta(delta)
    if start_date <= end_date:
        print('Warning: last aggregated table has less than {} days'.format(str(args.scale)))
        dates = '%s..%s' % (start_date.strftime('%Y-%m-%d'), end_date.strftime('%Y-%m-%d'))
        print 'Start calculation: ' + dates
        main(dates, args.scale, args.path)
